package filtering

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	wizardProto "a.yandex-team.ru/travel/proto/avia/wizard"
)

type (
	FaresProtoFilter struct {
		appLogger    log.Logger
		jobID        string
		flightsProto map[string]*wizardProto.Flight
		conditions   []FaresProtoFilterCondition
	}

	FaresProtoFilterCondition func(fare *wizardProto.Fare, index int) bool
)

func NewFaresProtoFilter(appLogger log.Logger, flightsProto map[string]*wizardProto.Flight, jobID string) *FaresProtoFilter {
	return &FaresProtoFilter{appLogger: appLogger, flightsProto: flightsProto, jobID: jobID}
}

func (filter *FaresProtoFilter) IsGoodFare(fare *wizardProto.Fare, index int) bool {
	for _, condition := range filter.conditions {
		if !condition(fare, index) {
			return false
		}
	}
	return true
}

func (filter *FaresProtoFilter) DepartsSoon(now time.Time, locationRepository repositories.CachedLocation) *FaresProtoFilter {
	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: departs soon", filter.jobID, index))
			}
		}()
		flight := filter.flightsProto[fare.Route.Forward[0]]
		location, err := locationRepository.LoadLocation(*flight.Departure.Tzname)
		if err != nil {
			return true
		}
		departureTime, err := helpers.ParseTimeInLocation(*flight.Departure.Local, location)
		if err != nil {
			return true
		}
		if *fare.Partner == consts.AeroflotPartnerCode {
			return !departureTime.Before(now.Add(consts.AeroflotBeforeDepartureMinTime))
		}

		return !departureTime.Before(now.Add(time.Hour))
	})
	return filter
}

func (filter *FaresProtoFilter) Expired(utcNow time.Time) *FaresProtoFilter {
	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: expired", filter.jobID, index))
			}
		}()
		return utcNow.Unix() <= int64(*fare.ExpireAt)
	})
	return filter
}

func (filter *FaresProtoFilter) PromoExpired(utcNow time.Time) *FaresProtoFilter {
	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: expired promo", filter.jobID, index))
			}
		}()

		if fare.Promo == nil || fare.Promo.Code == nil || *fare.Promo.Code == "" {
			return true
		}
		return fare.Promo.EndTs == nil || utcNow.Unix() <= int64(*fare.Promo.EndTs)
	})
	return filter
}

func (filter *FaresProtoFilter) DisabledPartner(partnerRepository repositories.Partner, nationalVersion string) *FaresProtoFilter {
	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d, disabled partner", filter.jobID, index))
			}
		}()
		return partnerRepository.IsEnabled(*fare.Partner, nationalVersion)
	})
	return filter
}

func (filter *FaresProtoFilter) IncorrectCompanyIDs(companyRepository repositories.Company) *FaresProtoFilter {
	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: incorrect company ID", filter.jobID, index))
			}
		}()
		for _, flightKey := range fare.Route.Forward {
			flight, ok := filter.flightsProto[flightKey]
			if !ok {
				return false
			}
			if flight.Company == nil {
				return false
			}
			if flight.Company != nil && *flight.Company <= 0 {
				return false
			}
			if _, found := companyRepository.GetByID(int(*flight.Company)); !found {
				return false
			}
		}
		for _, flightKey := range fare.Route.Backward {
			flight, ok := filter.flightsProto[flightKey]
			if !ok {
				return false
			}
			if flight.Company != nil && *flight.Company <= 0 {
				return false
			}
			if _, found := companyRepository.GetByID(int(*flight.Company)); !found {
				return false
			}
		}
		return true
	})
	return filter
}

func (filter *FaresProtoFilter) NoStationTranslations(
	stationRepository repositories.Station,
	translatedTitleRepository repositories.TranslatedTitle,
	lang models.Lang,
) *FaresProtoFilter {
	noStationTitleTranslation := func(stationTitleID int) bool {
		_, err := translatedTitleRepository.GetTitleTranslation(stationTitleID, lang, consts.CaseNominative)
		return err != nil
	}
	isTripCorrect := func(flightKeys []string) bool {
		for _, flightKey := range flightKeys {
			flight, ok := filter.flightsProto[flightKey]
			if !ok {
				return false
			}
			if stationFrom, ok := stationRepository.GetByID(int(*flight.FromId)); !ok || noStationTitleTranslation(stationFrom.NewLTitleID) {
				return false
			}
			if stationTo, ok := stationRepository.GetByID(int(*flight.ToId)); !ok || noStationTitleTranslation(stationTo.NewLTitleID) {
				return false
			}
		}
		return true
	}

	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: no stations translations", filter.jobID, index))
			}
		}()
		return isTripCorrect(fare.Route.Forward) && isTripCorrect(fare.Route.Backward)
	})
	return filter
}

func (filter *FaresProtoFilter) ShortStopover(locationRepository repositories.CachedLocation) *FaresProtoFilter {
	isShortStopover := func(arrivalTime, departureTme time.Time) bool {
		return departureTme.Sub(arrivalTime) < 5*time.Minute
	}
	tripHasShortStopovers := func(flightKeys []string) bool {
		var prevFlight *wizardProto.Flight
		for i, flightKey := range flightKeys {
			flight, ok := filter.flightsProto[flightKey]
			if !ok || flight.Arrival == nil || flight.Departure == nil {
				return true
			}
			if i == 0 {
				prevFlight = flight
				continue
			}
			prevLocation, err := locationRepository.LoadLocation(*prevFlight.Arrival.Tzname)
			if err != nil {
				return true
			}
			location, err := locationRepository.LoadLocation(*flight.Departure.Tzname)
			if err != nil {
				return true
			}
			departureTime, err := helpers.ParseTimeInLocation(*flight.Departure.Local, location)
			if err != nil {
				return true
			}
			arrivalTime, err := helpers.ParseTimeInLocation(*prevFlight.Arrival.Local, prevLocation)
			if err != nil || isShortStopover(arrivalTime, departureTime) {
				return true
			}
		}
		return false
	}

	filter.conditions = append(filter.conditions, func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: short stopover", filter.jobID, index))
			}
		}()
		return !(tripHasShortStopovers(fare.Route.Forward) || tripHasShortStopovers(fare.Route.Backward))
	})
	return filter
}

func (filter *FaresProtoFilter) ApplyUserFilters(locationRepository repositories.CachedLocation, filters dynamic.Filters) *FaresProtoFilter {
	if withBaggageValue, ok := filters.WithBaggage(); ok && withBaggageValue != nil && *withBaggageValue {
		filter.conditions = append(filter.conditions, filter.withBaggage)
	}
	if airlines, ok := filters.Airlines(); ok && !helpers.IsNil(airlines) && len(airlines) > 0 {
		filter.conditions = append(filter.conditions, filter.airlineFilter(airlines))
	}

	if transfer, ok := filters.Transfer(); ok && transfer != nil {
		if count, ok := transfer.Count(); ok && count != nil {
			filter.conditions = append(filter.conditions, filter.checkTransfersCount(*count))
		}
		if hasAirportChange, ok := transfer.HasAirportChange(); ok && hasAirportChange != nil {
			filter.conditions = append(filter.conditions, filter.noAirportChanges)
		}
		minDuration, minDurationOk := transfer.MinDuration()
		maxDuration, maxDurationOk := transfer.MaxDuration()
		if minDurationOk && minDuration != nil || maxDurationOk && maxDuration != nil {
			filter.conditions = append(filter.conditions, filter.checkTransferDuration(transfer, locationRepository))
		}
	}
	// TODO: RASPTICKETS-17580 Filter by airports
	// if filters.Airport != nil {
	// 	filter.conditions = append(filter.conditions, airportFilter(filters.Airport))
	// }

	// TODO: timeFilter
	return filter
}

func (filter *FaresProtoFilter) airlineFilter(airlineFilter containers.SetOfInt) FaresProtoFilterCondition {
	return func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: user's airline filter", filter.jobID, index))
			}
		}()
		for _, flightKey := range fare.Route.Forward {
			flight := filter.flightsProto[flightKey]
			if flight.Company == nil || !airlineFilter.Contains(int(*flight.Company)) {
				return false
			}
		}
		for _, flightKey := range fare.Route.Backward {
			flight := filter.flightsProto[flightKey]
			if flight.Company == nil || !airlineFilter.Contains(int(*flight.Company)) {
				return false
			}
		}
		return true
	}
}

func (filter *FaresProtoFilter) airportFilter(airportFilter *dynamic.AirportFilter) FaresProtoFilterCondition {
	return func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: user's airport filter", filter.jobID, index))
			}
		}()
		return !(filter.invalidEndpoints(fare.Route.Forward, airportFilter.ForwardDeparutre, airportFilter.ForwardArrival) ||
			filter.invalidEndpoints(fare.Route.Backward, airportFilter.BackwardDeparutre, airportFilter.BackwardArrival) ||
			filter.invalidTransfers(fare.Route.Forward, airportFilter.ForwardTransfers) ||
			filter.invalidTransfers(fare.Route.Backward, airportFilter.BackwardTransfers))
	}
}

func (filter *FaresProtoFilter) invalidTransfers(flightKeys []string, transfers containers.SetOfInt) bool {
	if len(transfers) == 0 {
		return false
	}
	transferAirports := make(containers.SetOfInt)
	for i, flightKey := range flightKeys {
		if i == 0 {
			continue
		}
		prevSegment := filter.flightsProto[flightKeys[i-1]]
		segment := filter.flightsProto[flightKey]
		transferAirports.Add(int(*prevSegment.ToId))
		transferAirports.Add(int(*segment.FromId))
	}
	return len(transfers.Intersect(transferAirports)) == 0
}

func (filter *FaresProtoFilter) invalidEndpoints(flightKeys []string, departures containers.SetOfInt, arrivals containers.SetOfInt) bool {
	if len(flightKeys) == 0 {
		return false
	}
	if len(departures) > 0 {
		if !departures.Contains(int(*filter.flightsProto[flightKeys[0]].FromId)) {
			return true
		}
	}
	if len(arrivals) > 0 {
		if !arrivals.Contains(int(*filter.flightsProto[flightKeys[len(flightKeys)-1]].ToId)) {
			return true
		}
	}
	return false
}

func (filter *FaresProtoFilter) checkTransferDuration(
	transferFilter dynamic.TransferFilter,
	locationRepository repositories.CachedLocation,
) FaresProtoFilterCondition {
	return func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: user's transfer duration filter", filter.jobID, index))
			}
		}()
		minDuration := 0
		if val, ok := transferFilter.MinDuration(); ok && val != nil {
			minDuration = *val
		}
		maxDuration := 60 * 24 * 7
		if val, ok := transferFilter.MaxDuration(); ok && val != nil {
			maxDuration = *val
		}
		checkTrip := func(flightKeys []string) bool {
			for i, flightKey := range flightKeys {
				if i == 0 {
					continue
				}
				prevFlight := filter.flightsProto[flightKeys[i-1]]
				flight := filter.flightsProto[flightKey]
				prevLocation, err := locationRepository.LoadLocation(*prevFlight.Arrival.Tzname)
				if err != nil {
					return false
				}
				location, err := locationRepository.LoadLocation(*flight.Departure.Tzname)
				if err != nil {
					return false
				}
				departureTime, err := helpers.ParseTimeInLocation(*flight.Departure.Local, location)
				if err != nil {
					return false
				}
				arrivalTime, err := helpers.ParseTimeInLocation(*prevFlight.Arrival.Local, prevLocation)
				if err != nil {
					return false
				}
				transferDuration := int(departureTime.Sub(arrivalTime).Minutes())
				if transferDuration <= minDuration || transferDuration >= maxDuration {
					return false
				}
				if hasNight, ok := transferFilter.HasNight(); ok && hasNight != nil {
					if arrivalTime.Day() != departureTime.Day() ||
						(0 <= arrivalTime.Hour() && arrivalTime.Hour() <= 6) ||
						(0 <= departureTime.Hour() && departureTime.Hour() <= 6) {
						return false
					}
				}
			}
			return true
		}
		return checkTrip(fare.Route.Forward) && checkTrip(fare.Route.Backward)
	}
}

func (filter *FaresProtoFilter) checkTransfersCount(transfersCount int) FaresProtoFilterCondition {
	return func(fare *wizardProto.Fare, index int) (result bool) {
		defer func() {
			if !result {
				filter.appLogger.Debug(fmt.Sprintf("job_id: %s: user's transfers count filter", filter.jobID))
			}
		}()
		return len(fare.Route.Forward) <= transfersCount+1 && len(fare.Route.Backward) <= transfersCount+1
	}
}

func (filter *FaresProtoFilter) noAirportChanges(fare *wizardProto.Fare, index int) (result bool) {
	defer func() {
		if !result {
			filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: no airport changes filter", filter.jobID, index))
		}
	}()
	for i, flightKey := range fare.Route.Forward {
		flight := filter.flightsProto[flightKey]
		if i > 0 {
			prevFlight := filter.flightsProto[fare.Route.Forward[i-1]]
			if prevFlight.ToId != flight.FromId {
				return false
			}
		}
	}
	for i, flightKey := range fare.Route.Backward {
		flight := filter.flightsProto[flightKey]
		if i > 0 {
			prevFlight := filter.flightsProto[fare.Route.Backward[i-1]]
			if prevFlight.ToId != flight.FromId {
				return false
			}
		}
	}
	return true
}

func (filter *FaresProtoFilter) withBaggage(fare *wizardProto.Fare, index int) (result bool) {
	defer func() {
		if !result {
			filter.appLogger.Debug(fmt.Sprintf("job_id: %s, index: %d: user's with baggage filter", filter.jobID, index))
		}
	}()
	included := true
	for _, baggage := range fare.Baggage.Forward {
		if baggage[0] != '1' {
			included = false
		}
	}
	for _, baggage := range fare.Baggage.Backward {
		if baggage[0] != '1' {
			included = false
		}
	}
	if included {
		return true
	} else if fare.Tariffs.WithBaggage != nil {
		fare.Partner = fare.Tariffs.WithBaggage.Partner
		fare.Tariff = fare.Tariffs.WithBaggage.Price
		fare.CreatedAt = fare.Tariffs.WithBaggage.CreatedAt
		fare.ExpireAt = fare.Tariffs.WithBaggage.ExpireAt
		fare.Baggage.Forward = fare.Tariffs.WithBaggage.Baggage.Forward
		fare.Baggage.Backward = fare.Tariffs.WithBaggage.Baggage.Forward
		return true
	}
	return false
}
