package flags

import (
	"math"
	"os"
	"reflect"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
)

type Flags map[string]string

func (flags *Flags) Equals(other *Flags) bool {
	return reflect.DeepEqual(flags, other)
}

func (flags Flags) IsEmpty() bool {
	return len(flags) == 0
}

func (flags *Flags) ReplaceAirportsByCities() bool {
	replaceAirportsByCities := flags.getOrDefault("replace_airports_by_cities", "1")
	return replaceAirportsByCities == "1"
}

func (flags *Flags) GeoIDAsToID() bool {
	replaceAirportsByCities := flags.getOrDefault("geo_id_as_to_id", "")
	return replaceAirportsByCities == "1"
}

func (flags *Flags) EnablePPCompanies() bool {
	replaceAirportsByCities := flags.getOrDefault("enable_pp_companies", "")
	return replaceAirportsByCities == "1"
}

func (flags *Flags) EnableSearchToAirports() bool {
	enableSearchToAirports := flags.getOrDefault("enable_search_to_airports", "")
	return enableSearchToAirports == "1"
}

func (flags *Flags) OffersInSideblock() int {
	offersUnderCut, _ := strconv.Atoi(flags.getOrDefault("offers_in_sideblock", "60"))
	return offersUnderCut
}

func (flags *Flags) EnablePPDirectFlight() bool {
	enableSearchToAirports := flags.getOrDefault("enable_direct_filter", "")
	return enableSearchToAirports == "1"
}

func (flags *Flags) EnableQasiFlightValues() containers.SetOfString {
	return containers.NewSetOfStringFromSlices(
		strings.Split(
			flags.getOrDefault("enable_quasi_flight", ""), ":",
		),
	)
}

func (flags Flags) ChangeByToRu() bool {
	return flags.getOrDefault("change_by_to_ru", "1") == "1"
}

func (flags Flags) GeoIDLookup() bool {
	return flags.getOrDefault("geoid_lookup", "1") == "1"
}

func (flags Flags) getOrDefault(key, defaultValue string) string {
	if value, found := flags[key]; found {
		return value
	}
	return defaultValue
}

func (flags *Flags) FallbackAirlineFilter() bool {
	return flags.getOrDefault("fallback_airline_filter", "") == "1"
}

func (flags *Flags) FallbackDirectFilter() bool {
	return flags.getOrDefault("fallback_airline_filter", "") == "1"
}

func (flags *Flags) ButtonRedirectToSearchResultPage() bool {
	return flags.getOrDefault("button_redirect_to_search_result_page", "") == "1"
}

func (flags *Flags) ButtonRedirectToOrderPage() bool {
	return flags.getOrDefault("button_redirect_to_order_page", "") == "1"
}

func (flags *Flags) AntiFASOfferRedirects() bool {
	return flags.getOrDefault("avia_antifas_order_redirects", "") == "1"
}

func (flags *Flags) SearchDepthButtonRedirectToOrderPage() int {
	value, err := strconv.Atoi(flags.getOrDefault("search_depth_button_redirect_to_order_page", ""))
	if err != nil {
		return math.MaxInt64
	}
	return value
}

func (flags *Flags) SnippetRedirectToSearchResultPage() bool {
	return flags.getOrDefault("snippet_redirect_to_search_result_page", "") == "1"
}

func (flags *Flags) KKEnabled() bool {
	return flags.getOrDefault("kk_enabled", "") == "1"
}

func (flags *Flags) SleepPPStaticDuration() time.Duration {
	return parseSleepDuration(flags.getOrDefault("sleep_pp_static", "0"))
}

func (flags *Flags) SleepPPDynamicDuration() time.Duration {
	return parseSleepDuration(flags.getOrDefault("sleep_pp_dynamic", "0"))
}

func (flags *Flags) SearchDepthExperiment() bool {
	return flags.getOrDefault("search_depth_experiment", "") == "1"
}

func (flags *Flags) EnableStaticExperimentalQueries() bool {
	return flags.getOrDefault("enable_static_experimental_queries", "") == "1"
}

func (flags *Flags) UseTTLParallelRequests() bool {
	return flags.getOrDefault("use_ttl_parallel_requests", "") == "1"
}

func (flags *Flags) SortByPrice() bool {
	return flags.getOrDefault("sort_by_price", "") == "1"
}

func (flags *Flags) PersonalizedDefaultDay() bool {
	return flags.getOrDefault("personalized_default_day", "") == "1"
}

func (flags *Flags) PersonalizedDefaultDayOrMinPriceDate() bool {
	return flags.getOrDefault("personalized_default_day_or_min_price_date", "1") == "1"
}

func parseSleepDuration(rawSleepDuration string) time.Duration {
	sleepSeconds, err := strconv.ParseFloat(rawSleepDuration, 64)
	if err != nil {
		return 0
	}

	// Отбрасываем отрицательные значения, а также не спим дольше 10 секунд
	sleepSeconds = math.Max(sleepSeconds, 0)
	sleepSeconds = math.Min(sleepSeconds, 10)

	return time.Duration(sleepSeconds * float64(time.Second))
}

func (flags *Flags) DisableDeprecatedFlightResponseFormat() bool {
	return flags.getOrDefault("disable_deprecated_flight_response_format", "") == "1"
}

func (flags *Flags) UseXRedirect() bool {
	return os.Getenv("AVIA_WIZARD_USE_XREDIRECT") == "1"
}

func (flags *Flags) UseUnisearch() bool {
	return os.Getenv("AVIA_WIZARD_IS_UNISEARCH") == "1"
}

func (flags *Flags) EnablePartnerCache() bool {
	return flags.getOrDefault("enable_partner_cache", "") == "1"
}

func (flags *Flags) EnableNewPartnerSort() bool {
	return flags.getOrDefault("enable_new_partner_sort", "") == "1"
}

func (flags *Flags) EnablePrimaryPartnerFreeRedirect() bool {
	return os.Getenv("AVIA_WIZARD_PRIMARY_PARTNER_FREE_REDIRECT") == "1"
}

func (flags *Flags) EnableTutuPartnerResults() bool {
	return flags.getOrDefault("enable_tutu", "") == "1"
}
