package flags

import (
	"fmt"
	"strings"
)

type Parser interface {
	Parse(raw string) (Flags, error)
}

type RawStringParser struct {
}

func NewParser() Parser {
	return new(RawStringParser)
}

func (parser *RawStringParser) parseFlag(rawFlag string) (key string, value string, err error) {
	if len(rawFlag) == 0 {
		return "", "", NewFlagParsingError(rawFlag)
	}

	equalSignsCount := strings.Count(rawFlag, "=")
	if equalSignsCount == 0 {
		return rawFlag, "", nil
	}

	if equalSignsCount > 1 {
		return "", "", NewFlagParsingError(rawFlag)
	}

	equalSignIndex := strings.Index(rawFlag, "=")
	if equalSignIndex == 0 {
		return "", "", NewFlagParsingError(rawFlag)
	}

	return rawFlag[0:equalSignIndex], rawFlag[equalSignIndex+1:], nil
}

func (parser *RawStringParser) Parse(raw string) (Flags, error) {
	if len(raw) == 0 {
		return make(Flags), nil
	}

	flags := make(Flags, strings.Count(raw, ",")+1)

	for _, rawFlag := range strings.Split(raw, ",") {
		key, value, err := parser.parseFlag(rawFlag)
		if err != nil {
			return nil, err
		}
		flags[key] = value
	}

	return flags, nil
}

type FlagParsingError struct {
	RawFlagValue string
}

func NewFlagParsingError(rawFlagValue string) *FlagParsingError {
	return &FlagParsingError{rawFlagValue}
}

func (err *FlagParsingError) Error() string {
	return fmt.Sprintf("Flag has wrong format: %s", err.RawFlagValue)
}
