package building

import (
	"context"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type CommonResponseBuilder struct {
	landingBuilder            landings.FrontLandingBuilder
	commonTranslator          *translations.CommonTranslator
	translatedTitleRepository repositories.TranslatedTitle
}

func NewCommonResponseBuilder(
	landingBuilder landings.FrontLandingBuilder,
	commonTranslator *translations.CommonTranslator,
	translatedTitleRepository repositories.TranslatedTitle,
) *CommonResponseBuilder {
	return &CommonResponseBuilder{
		landingBuilder:            landingBuilder,
		commonTranslator:          commonTranslator,
		translatedTitleRepository: translatedTitleRepository,
	}
}

func (commonResponseBuilder *CommonResponseBuilder) Build(
	queryParameters *parameters.QueryParameters,
	landingParameters map[string]string,
	fromSettlement *models.Settlement,
	ctx context.Context,
	departureDate *string,
) (responses.HandlerResponse, error) {
	if helpers.IsNil(fromSettlement) {
		return nil, domain.NewWizardError("couldn't parse fromSettlement for common wizard", domain.BadPoint)
	}
	wizardType := WizardTypeCommon
	landingBuilder := commonResponseBuilder.landingBuilder.WithQueryParameters(queryParameters)
	if queryParameters.Company != nil {
		wizardType = WizardTypeCommonAirline
	}
	props.SetSearchProp(ctx, "wizard_type", wizardType.String())
	fromSettlementTitle, err := commonResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		fromSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	if err != nil {
		return nil, domain.NewWizardError(err.Error(), domain.NoTranslation)
	}

	useTravelPortal := consts.TravelPortalUsage(helpers.IsNil(queryParameters.Company)) && queryParameters.UseTravelPortal()
	greenURLs := []responses.URL{
		{
			Text: commonResponseBuilder.commonTranslator.LocServiceName(queryParameters.Lang, useTravelPortal),
			URL: landingBuilder.
				Morda(useTravelPortal).
				WithParams(landingParameters).
				WithParam("utm_content", "greenurl_1").
				AsString(),
		},
	}
	title := responses.Title{}

	var landing *landings.Landing
	company := queryParameters.Company
	if company != nil {
		landing = landingBuilder.AirlineFromSettlement(company, fromSettlement).WithParams(landingParameters)
		companyTitle, err := commonResponseBuilder.translatedTitleRepository.GetOldTitleTranslation(company, queryParameters.Lang)
		if err != nil {
			return nil, domain.NewWizardError(err.Error(), domain.NoTranslation)
		}
		greenURLs = append(
			greenURLs, responses.URL{
				Text: commonResponseBuilder.commonTranslator.LocAirlineRoutes(companyTitle, queryParameters.Lang),
				URL:  landing.WithParam("utm_content", "greenurl_2").AsString(),
			},
		)
		title.URL = landing.WithParam("utm_content", "title").AsString()
		commonAirlineTitle, err := commonResponseBuilder.commonTranslator.LocCommonAirlineTitle(
			companyTitle,
			fromSettlement,
			queryParameters.Lang,
		)
		if err != nil {
			return nil, domain.NewWizardError(err.Error(), domain.NoTranslation)
		}
		title.Text.HL = commonAirlineTitle
	} else {
		landing = landingBuilder.
			City(fromSettlementTitle, fromSettlement.IataCode, fromSettlement.GetPointKey()).
			WithParams(landingParameters)
		greenURL2Text, err := commonResponseBuilder.commonTranslator.LocCommonTicketsFrom(fromSettlement, queryParameters.Lang)
		if err != nil {
			return nil, domain.NewWizardError(err.Error(), domain.NoTranslation)
		}
		greenURLs = append(
			greenURLs, responses.URL{
				Text: greenURL2Text,
				URL:  landing.WithParam("utm_content", "greenurl_2").AsString(),
			},
		)
		title.URL = landing.WithParam("utm_content", "title").AsString()
		commonTitle, err := commonResponseBuilder.commonTranslator.LocCommonTitle(
			fromSettlement,
			queryParameters.Lang,
			queryParameters.UseTravelPortal(),
			queryParameters.Flags.AntiFASOfferRedirects(),
		)
		if err != nil {
			return nil, domain.NewWizardError(err.Error(), domain.NoTranslation)
		}
		title.Text.HL = commonTitle
	}

	offers := responses.Offers{
		TitleTariff: nil,
		Variants:    make([]*responses.Variant, 0),
		Total:       0,
		ReturnDate:  nil,
		Sideblock:   nil,
	}

	if departureDate != nil {
		offers.DepartureDate = *departureDate
	}
	response := &responses.CommonResponse{
		GreenURL: greenURLs,
		Context:  nil,
		Title:    title,
		Button: responses.URL{
			Text: commonResponseBuilder.commonTranslator.LocButton(queryParameters.Lang),
			URL:  landing.WithParam("utm_content", "more_offers").AsString(),
		},
		Factors: responses.Factors{},
		Flags:   make(map[string]string),
		Content: responses.CommonContent{
			Text:   commonResponseBuilder.commonTranslator.LocCommonWizardText(queryParameters.Lang),
			Offers: offers,
		},
		From: responses.Point{
			PointKey: fromSettlement.GetPointKey(),
			ID:       fromSettlement.ID,
			Title:    fromSettlementTitle,
		},
		Reqid:          queryParameters.MainReqID,
		SearchProps:    nil,
		Companies:      make([]int, 0),
		Directions:     make([]interface{}, 0),
		Params:         landingParameters,
		AppliedFilters: commonResponseBuilder.buildAppliedFilters(queryParameters, company),
		Type:           wizardType.String(),
	}
	if company != nil {
		response.Factors.TypeCommonAirline = ptr.Int(1)
		response.Factors.Airline = ptr.Int(1)
		response.QueryCompany = &responses.QueryCompany{ID: company.ID}
	} else {
		response.Factors.TypeCommon = ptr.Int(1)
		response.Factors.Airline = ptr.Int(0)
	}

	if queryParameters.Flags.EnablePrimaryPartnerFreeRedirect() {
		if partnerCode := queryParameters.PartnerCode; partnerCode != nil {
			response.PrimaryPartnerCode = *partnerCode
		}
	}
	return response, nil
}

func (commonResponseBuilder *CommonResponseBuilder) buildAppliedFilters(
	queryParameters *parameters.QueryParameters,
	company *models.Company,
) *responses.AppliedFilters {
	if company == nil && (!queryParameters.DirectFlight || !queryParameters.Flags.EnablePPDirectFlight()) {
		return &responses.AppliedFilters{}
	}
	appliedFilters := responses.AppliedFilters{}
	if company != nil {
		airlines := containers.NewSetOfInt(company.ID)
		appliedFilters.Airlines = &airlines
	}
	if queryParameters.PartnerCode != nil && *queryParameters.PartnerCode != "" {
		partners := containers.NewSetOfString(*queryParameters.PartnerCode)
		appliedFilters.Partners = &partners
	}
	if queryParameters.DirectFlight && queryParameters.Flags.EnablePPDirectFlight() {
		appliedFilters.Transfer = &responses.TransferFilter{Count: ptr.Int(0)}
	}
	return &appliedFilters
}
