package building

import (
	"fmt"
	"math"
	"regexp"
	"sort"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/ptr"
	flightsModel "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type (
	FlightResponseBuilder struct {
		flightTranslator          *translations.FlightTranslator
		timeTranslator            *translations.TimeTranslator
		commonTranslator          *translations.CommonTranslator
		settlementRepository      repositories.Settlement
		stationRepository         repositories.Station
		translatedTitleRepository repositories.TranslatedTitle
		landingBuilder            landings.FrontLandingBuilder
	}

	FlightResponseContext struct {
		tabs               Tabs
		landingBuilder     landings.FrontLandingBuilder
		fromPoint, toPoint models.Point
		flightNumber       string
		nationalVersion    string
		device             string
		language           models.Lang
		nowAware           time.Time
		defaultDate        time.Time
		defaultFlight      *flightsModel.Flight
	}
)

func NewFlightResultBuilder(
	flightTranslator *translations.FlightTranslator,
	timeTranslator *translations.TimeTranslator,
	commonTranslator *translations.CommonTranslator,
	settlementRepository repositories.Settlement,
	translatedTitleRepository repositories.TranslatedTitle,
	landingBuilder landings.FrontLandingBuilder,
	stationRepository repositories.Station,
) FlightResponseBuilder {
	return FlightResponseBuilder{
		flightTranslator:          flightTranslator,
		timeTranslator:            timeTranslator,
		commonTranslator:          commonTranslator,
		settlementRepository:      settlementRepository,
		translatedTitleRepository: translatedTitleRepository,
		landingBuilder:            landingBuilder,
		stationRepository:         stationRepository,
	}
}

func (builder *FlightResponseBuilder) Build(tabs Tabs, fromPoint, toPoint models.Point, flightNumber string, queryParameters *parameters.QueryParameters, nowAware, defaultDate time.Time, defaultFlight *flightsModel.Flight, landingParams map[string]string, companyCode string) (*responses.FlightVariant, error) {
	context := FlightResponseContext{
		tabs:            tabs,
		landingBuilder:  builder.landingBuilder.WithQueryParameters(queryParameters),
		fromPoint:       fromPoint,
		toPoint:         toPoint,
		flightNumber:    flightNumber,
		nationalVersion: queryParameters.NationalVersion,
		device:          queryParameters.Device,
		language:        queryParameters.Lang,
		nowAware:        nowAware,
		defaultDate:     defaultDate,
		defaultFlight:   defaultFlight,
	}

	flightLanding := context.landingBuilder.Flight(context.flightNumber, &context.defaultDate).WithParams(landingParams)

	content, err := builder.getTabContent(context, landingParams)
	if err != nil {
		return nil, err
	}
	titleText, err := builder.flightTranslator.LocTitle(
		context.flightNumber,
		context.language,
		&context.defaultFlight.Company,
		context.fromPoint,
		context.toPoint,
	)
	if err != nil {
		return nil, err
	}
	typeFlight := 1

	seeMoreText, err := builder.flightTranslator.LocSeeMore(context.language)
	if err != nil {
		return nil, err
	}

	return &responses.FlightVariant{
		Content: content,
		Title: responses.Title{
			Text: responses.TitleText{HL: titleText},
			URL: flightLanding.
				WithParam("utm_content", "title").
				AsString(),
		},

		Button: responses.URL{
			Text: seeMoreText,
			URL: flightLanding.
				WithParam("utm_content", "flightpage_link").
				AsString(),
		},

		GreenURL: []responses.URL{
			{
				Text: builder.commonTranslator.LocServiceName(context.language, queryParameters.UseTravelPortal()),
				URL: context.landingBuilder.
					FlightMorda().
					WithParams(landingParams).
					WithParam("utm_content", "greenurl_1").
					AsString(),
			},
			{
				Text: builder.getFromToGreenURLText(context),
				URL: context.landingBuilder.
					FlightRoutes(
						context.defaultFlight.PointFrom.AirportCode, context.defaultFlight.PointTo.AirportCode,
						context.fromPoint, context.toPoint,
						context.language,
					).
					WithParams(landingParams).
					WithParam("utm_content", "greenurl_2").
					AsString(),
			},
			{
				Text: context.flightNumber,
				URL: flightLanding.
					WithParam("utm_content", "greenurl_3").
					AsString(),
			},
		},
		Factors:     responses.Factors{TypeFlight: &typeFlight},
		CompanyCode: companyCode,
	}, nil

}

func (builder *FlightResponseBuilder) getTabContent(
	context FlightResponseContext,
	landingParams map[string]string,
) (responses.FlightContent, error) {
	text, err := builder.flightTranslator.LocContentText(
		context.flightNumber,
		context.language,
		&context.defaultFlight.Company,
		context.fromPoint,
		context.toPoint,
	)
	if err != nil {
		return responses.FlightContent{}, err
	}

	content := responses.FlightContent{Text: text}

	for _, tab := range context.tabs {
		tabFlights := make([]responses.Flight, 0, len(tab.Flights))
		for _, flight := range tab.Flights {
			flightResult, err := builder.getFlightAsResult(context, flight, landingParams)

			if err != nil {
				return responses.FlightContent{}, err
			}
			tabFlights = append(tabFlights, flightResult)
		}
		title, err := builder.timeTranslator.LocMonthDayWeekday(tab.Date, context.language)
		if err != nil {
			return responses.FlightContent{}, err
		}

		content.Tabs = append(content.Tabs, responses.Tab{
			Title:   title,
			Default: tab.Date.Equal(context.defaultDate),
			Flights: tabFlights,
		})
	}
	return content, nil
}

func (builder *FlightResponseBuilder) getFlightAsResult(
	context FlightResponseContext,
	flight *flightsModel.Flight,
	landingParams map[string]string,
) (flightData responses.Flight, err error) {
	duration := flight.PointTo.ScheduledTime.Sub(flight.PointFrom.ScheduledTime)
	if duration < 10*time.Minute {
		return flightData, ErrBadFlight
	}

	inAirDudation, passed := builder.getFlightProgress(context, flight, duration)
	var inAir *string
	if inAirDudation != nil {
		inAir = ptr.String(builder.flightTranslator.LocDuration(*inAirDudation, context.language))
	}
	var lastUpdate *string
	departsSoon := context.nowAware.After(flight.PointFrom.GetEstimatedTime().Add(-24 * time.Hour))
	if departsSoon {
		lastUpdateString, err := builder.flightTranslator.LocLastUpdate(
			context.nowAware.Sub(flight.UpdatedAt), context.language,
		)
		if err != nil {
			return responses.Flight{}, err
		}
		lastUpdate = &lastUpdateString
	}

	seeMoreText, err := builder.flightTranslator.LocSeeMore(context.language)
	if err != nil {
		return responses.Flight{}, err
	}

	status, err := builder.getFlightStatusResponse(context, *flight)
	if err != nil {
		return responses.Flight{}, err
	}

	return responses.Flight{
		Number:    flight.Number,
		Departure: builder.getFlightPointAsResult(context, flight.PointFrom, flight.StatusInfo, context.device),
		Arrival:   builder.getFlightPointAsResult(context, flight.PointTo, flight.StatusInfo, context.device),
		Duration: responses.Duration{
			InAir:  inAir,
			Total:  builder.flightTranslator.LocDuration(duration, context.language),
			Passed: passed,
		},
		LastUpdate: lastUpdate,
		Buttons: []responses.URL{
			{
				Text: seeMoreText,
				URL: context.landingBuilder.
					Flight(flight.Number, &flight.PointFrom.ScheduledTime).
					WithParams(landingParams).
					WithParam("utm_content", "flightpage_link").
					AsString(),
			},
		},
		Status: status,
	}, nil
}

func (builder *FlightResponseBuilder) getFlightProgress(
	context FlightResponseContext,
	flight *flightsModel.Flight,
	duration time.Duration,
) (_ *time.Duration, passed float64) {
	if !flight.PointFrom.IsActual || !flight.PointTo.IsActual {
		return nil, passed
	}
	if flight.StatusInfo.Status == "unknown" {
		return nil, passed
	}
	if flight.PointFrom.Time != nil && context.nowAware.Before(*flight.PointFrom.Time) {
		return nil, passed
	}

	if flight.PointTo.Time != nil && context.nowAware.Before(*flight.PointTo.Time) {
		inAirDuration := context.nowAware.Sub(*flight.PointFrom.Time)
		passed = math.Min(float64(inAirDuration)/float64(duration), 1)
		return &inAirDuration, passed
	}

	return nil, 1
}

func (builder *FlightResponseBuilder) getFlightPointAsResult(
	context FlightResponseContext,
	point flightsModel.Point,
	info flightsModel.StatusInfo,
	device string,
) responses.FlightPoint {
	airport, _ := builder.translatedTitleRepository.GetTitleTranslation(
		point.Airport.NewLTitleID, context.language, consts.CaseNominative)

	city := airport
	if settlement, found := builder.settlementRepository.GetByID(point.Airport.SettlementID); found {
		city, _ = builder.translatedTitleRepository.GetTitleTranslation(
			settlement.NewLTitleID, context.language, consts.CaseNominative)
	}

	actualTime := ""
	if point.Time != nil && !info.Cancelled() {
		actualTime = point.Time.Format("2006-01-02 15:04:05")
	}

	pointColors := builder.getColoredFlightPointResult(point)

	locDate, _ := builder.timeTranslator.LocMonthDay(point.GetEstimatedTime(), context.language)

	gateTTL := point.GetEstimatedTime().Add(20 * time.Minute)
	gate := formatGate(point.Gate, device)
	if context.nowAware.After(gateTTL) {
		gate = nil
	}

	return responses.FlightPoint{
		City:          city,
		Airport:       airport,
		Iata:          point.AirportCode,
		ScheduledTime: point.ScheduledTime.Format("2006-01-02 15:04:05"),
		Time:          actualTime,
		Terminal:      point.Terminal,
		Gate:          gate,
		FullGate:      point.Gate,
		Date:          locDate,
		DataSource:    point.DataSource,

		TerminalColor: pointColors.TerminalColor,
		GateColor:     pointColors.GateColor,
		TimeColor:     pointColors.TimeColor,
		DateColor:     pointColors.DateColor,
	}
}

var (
	gatePattern = regexp.MustCompile(`[,.;]`)
)

func formatGate(fullGate *string, device string) *string {
	if device != "touch" {
		return fullGate
	}
	if fullGate == nil || len(*fullGate) == 0 {
		return fullGate
	}
	gates := gatePattern.Split(*fullGate, -1)
	nonEmptyGates := []string{}
	for _, g := range gates {
		if len(g) > 0 {
			nonEmptyGates = append(nonEmptyGates, g)
		}
	}
	if len(nonEmptyGates) == 0 {
		return fullGate
	}
	sort.SliceStable(nonEmptyGates, func(i, j int) bool {
		return len(nonEmptyGates[i]) < len(nonEmptyGates[j])
	})
	shortestGateLength := len(nonEmptyGates[0])
	result := nonEmptyGates[0]
	if len(result) > 4 {
		result = result[:4]
	}
	if shortestGateLength > 4 || len(nonEmptyGates) > 1 {
		result += "..."
	}
	return &result
}

func formatCheckInDesks(fullCheckInDesks *string) *string {
	maxShownSymbols := 7
	if helpers.IsNil(fullCheckInDesks) {
		return fullCheckInDesks
	}
	intervals := strings.Split(*fullCheckInDesks, ",")
	if len(intervals[0]) >= maxShownSymbols {
		return ptr.String(intervals[0][:maxShownSymbols])
	}
	if len(*fullCheckInDesks) > maxShownSymbols {
		*fullCheckInDesks = (*fullCheckInDesks)[:maxShownSymbols+1]
	}
	lastComma := strings.LastIndex(*fullCheckInDesks, ",")
	if lastComma > 0 {
		return ptr.String((*fullCheckInDesks)[:lastComma])
	}
	return ptr.String(intervals[0])
}

var textColorByStatus = map[string]consts.Color{
	"unknown":   consts.Yellow,
	"cancelled": consts.Red,
	"arrived":   consts.Grey,
	"delayed":   consts.Red,
	"early":     consts.Red,
	"on_time":   consts.Green,
}

func (builder *FlightResponseBuilder) getFlightStatusResponse(
	context FlightResponseContext,
	flight flightsModel.Flight,
) (responses.Status, error) {
	status := flight.StatusInfo

	ttlDesksInfo := flight.PointFrom.GetEstimatedTime()
	isCheckInDesksInfoActual := !context.nowAware.After(ttlDesksInfo)

	var checkInDesks *string
	if isCheckInDesksInfoActual {
		checkInDesks = helpers.OptionalString(status.CheckInDesks)
	}

	startShowingBaggageCarousels := flight.PointTo.GetEstimatedTime().Add(-20 * time.Minute)
	isBaggageCarouselsInfoActual := context.nowAware.After(flight.PointFrom.GetEstimatedTime().Add(20*time.Minute)) &&
		!context.nowAware.Before(startShowingBaggageCarousels)
	var baggageCarousels *string
	if isBaggageCarouselsInfoActual {
		baggageCarousels = helpers.OptionalString(status.BaggageCarousels)
	}

	textColor := consts.Grey
	if color, found := textColorByStatus[status.Status]; found {
		textColor = color
	}

	statusText, err := builder.getFlightStatusText(flight, context.language)
	if err != nil {
		return responses.Status{}, err
	}

	return responses.Status{
		Text:             statusText,
		Code:             status.Status,
		DepartsSoon:      context.nowAware.After(flight.PointFrom.GetEstimatedTime().Add(-24 * time.Hour)),
		CheckInDesks:     formatCheckInDesks(checkInDesks),
		BaggageCarousels: baggageCarousels,

		TextColor:             string(textColor),
		CheckInDesksColor:     string(consts.Grey),
		BaggageCarouselsColor: string(consts.Grey),
	}, nil
}

func (builder *FlightResponseBuilder) getFlightStatusText(
	flight flightsModel.Flight,
	language models.Lang,
) (string, error) {
	if flight.StatusInfo.Diverted && flight.StatusInfo.DivertedAirportID != 0 {
		if airport, ok := builder.stationRepository.GetByID(flight.StatusInfo.DivertedAirportID); ok {
			if code, codeExists := builder.stationRepository.GetAnyCodeByID(airport.ID); codeExists {
				return builder.flightTranslator.LocFlightStatusDiverted(airport, code, language)
			}
		}
	}

	return builder.flightTranslator.LocFlightStatus(
		flight.StatusInfo.Status,
		flight.StatusInfo.DepartureIsKnown,
		flight.StatusInfo.ArrivalIsKnown,
		language,
	)
}

type pointColors struct {
	TerminalColor string
	GateColor     string
	TimeColor     string
	DateColor     string
}

func (builder *FlightResponseBuilder) getColoredFlightPointResult(
	point flightsModel.Point,
) pointColors {
	timeColorChooser := func() consts.Color {
		if point.Time == nil {
			return consts.Grey
		}
		if point.Time.After(point.ScheduledTime) {
			return consts.Red
		}
		if point.Time.Equal(point.ScheduledTime) {
			return consts.Black
		}
		return consts.Grey
	}

	dateColorChooser := func() consts.Color {
		if point.Time == nil {
			return consts.Grey
		}
		pointTime := helpers.TruncateToDate(*point.Time)
		pointScheduledTime := helpers.TruncateToDate(point.ScheduledTime)

		if pointTime.After(pointScheduledTime) {
			return consts.Red
		}
		if pointTime.Equal(pointScheduledTime) {
			return consts.Black
		}
		return consts.Grey
	}

	return pointColors{
		TerminalColor: string(consts.Grey),
		GateColor:     string(consts.Grey),
		TimeColor:     string(timeColorChooser()),
		DateColor:     string(dateColorChooser()),
	}
}

func (builder *FlightResponseBuilder) getFromToGreenURLText(context FlightResponseContext) string {
	getTitle := builder.translatedTitleRepository.GetTitleTranslation

	fromTitle, _ := getTitle(context.fromPoint.GetTitleID(), context.language, consts.CaseNominative)
	toTitle, _ := getTitle(context.toPoint.GetTitleID(), context.language, consts.CaseNominative)

	return fmt.Sprintf("%s%s%s %s", fromTitle, consts.NBSP, consts.ENDASH, toTitle)
}
