package building

import (
	"context"
	"fmt"
	"math"
	"strconv"
	"strings"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point/providers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type PointToPointResponseBuilder struct {
	variantsBuilder               *VariantsBuilder
	translatedTitleRepository     repositories.TranslatedTitle
	landingBuilder                landings.FrontLandingBuilder
	companyRepository             repositories.Company
	stationRepository             repositories.Station
	pointToPointTranslator        *translations.PointToPointTranslator
	commonTranslator              *translations.CommonTranslator
	nearestAviaSettlementProvider *providers.NearestAviaSettlementProvider
	directionRepository           repositories.Direction
	partnerRepository             repositories.Partner
	minPriceRepository            repositories.MinPrices
	currencyRepository            repositories.Currency
	routeCountRepository          repositories.RouteCount
	logger                        log.Logger
}

func NewPointToPointResponseBuilder(
	variantsBuilder *VariantsBuilder,
	translatedTitleRepository repositories.TranslatedTitle,
	companyRepository repositories.Company,
	stationRepository repositories.Station,
	landingBuilder landings.FrontLandingBuilder,
	pointToPointTranslator *translations.PointToPointTranslator,
	commonTranslator *translations.CommonTranslator,
	nearestAviaSettlementProvider *providers.NearestAviaSettlementProvider,
	directionRepository repositories.Direction,
	partnerRepository repositories.Partner,
	minPriceRepository repositories.MinPrices,
	currencyRepository repositories.Currency,
	routeCountRepository repositories.RouteCount,
	logger log.Logger,
) *PointToPointResponseBuilder {
	return &PointToPointResponseBuilder{
		variantsBuilder:               variantsBuilder,
		translatedTitleRepository:     translatedTitleRepository,
		landingBuilder:                landingBuilder,
		companyRepository:             companyRepository,
		stationRepository:             stationRepository,
		pointToPointTranslator:        pointToPointTranslator,
		commonTranslator:              commonTranslator,
		nearestAviaSettlementProvider: nearestAviaSettlementProvider,
		directionRepository:           directionRepository,
		partnerRepository:             partnerRepository,
		minPriceRepository:            minPriceRepository,
		currencyRepository:            currencyRepository,
		routeCountRepository:          routeCountRepository,
		logger:                        logger,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) Build(
	ctx context.Context,
	searchResult *results.SearchResult,
	handlerType, utmCampaign string,
	wizardType WizardType,
	queryParameters *parameters.QueryParameters,
	landingParameters map[string]string,
	fromSettlement, toSettlement *models.Settlement,
	toCrimea bool,
	boyIsEnabled bool,
) (*responses.PointToPointResponse, error) {
	now := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().ResponseBuildingTimer.RecordDuration(time.Since(now))
	}()
	span, ctx := opentracing.StartSpanFromContext(ctx, "Building point-to-point result")
	defer span.Finish()
	landingBuilder := ppResponseBuilder.landingBuilder.WithQueryParameters(queryParameters)

	fromName, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		fromSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	toName, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		toSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	passengers := queryParameters.Passengers()
	searchResultLanding := landingBuilder.SearchResult(
		fromSettlement, toSettlement,
		fromName, toName,
		searchResult.DateForward, searchResult.DateBackward,
		&passengers,
	).WithParams(landingParameters)
	searchDepth := int(searchResult.DateForward.Sub(helpers.TruncateToDate(time.Now())).Seconds()) / consts.SecondsInDay
	variants := ppResponseBuilder.variantsBuilder.buildVariants(
		ctx,
		searchResult.FareGenerator,
		queryParameters,
		fromSettlement,
		toSettlement,
		landingParameters,
		searchResultLanding,
		&landingBuilder,
		searchDepth,
	)

	currency := ppResponseBuilder.currencyRepository.GetByCode(consts.GetCurrencyCodeByNationalVersion(queryParameters.NationalVersion))
	prices := ppResponseBuilder.minPriceRepository.GetPricesByDirection(
		fromSettlement.ID,
		toSettlement.ID,
		currency.ID,
		queryParameters.NationalVersion,
	)
	if helpers.IsUkraine(fromSettlement, toSettlement) {
		prices = []models.ForwardDayPrice{}
	}
	routesLanding := ppResponseBuilder.setMinDateMaxDate(
		landingBuilder.Routes(
			fromSettlement, toSettlement,
			queryParameters.Lang,
			&passengers,
			toCrimea,
		).WithParams(landingParameters),
		variants,
		prices,
	)

	fromSettlementTitle, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		fromSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	toSettlementTitle, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		toSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	if toCrimea {
		toSettlementTitle = ppResponseBuilder.translatedTitleRepository.GetCrimeaTranslation(queryParameters.Lang)
	}
	appliedFilters := ppResponseBuilder.buildAppliedFilters(queryParameters)
	pollingStatus := ppResponseBuilder.buildPollingStatus(searchResult.PollingStatus, queryParameters, boyIsEnabled)
	totalOffers := searchResult.TotalFaresCount
	noVariants := len(variants) == 0
	go ppResponseBuilder.writeStats(queryParameters, searchResult.IsCacheMiss, noVariants)
	if noVariants && !ppResponseBuilder.hasAnyAppliedFilters(appliedFilters) {
		totalOffers = 0
	}
	nv := queryParameters.NationalVersion
	if nv == "by" {
		nv = "ru"
	}
	if noVariants && !ppResponseBuilder.directionRepository.IsPossibleVariant(fromSettlement.ID, toSettlement.ID, nv) {
		ppResponseBuilder.logger.Info(
			fmt.Sprintf(
				"no prices for main_reqid: %s, reqid: %s", queryParameters.MainReqID,
				queryParameters.ReqID,
			), log.String("job_id", queryParameters.JobID),
		)
		return nil, domain.NewWizardError("No prices found", domain.NoContent)
	}

	offers := &responses.Offers{
		Cut:           queryParameters.Context().OffersUnderCut(),
		Show:          queryParameters.Context().OffersToShow(),
		DepartureDate: searchResult.DateForward.Format("2006-01-02"),
		TitleTariff:   ppResponseBuilder.buildTitleTariff(variants, prices, searchResult.Filters, currency),
		Variants:      variants,
		Total:         totalOffers,
	}
	ppResponseBuilder.setReturnDate(searchResult, offers, queryParameters)
	if queryParameters.IsDynamic() && queryParameters.AviaDynamic.Sideblock != nil && *queryParameters.AviaDynamic.Sideblock != 0 {
		offers.Sideblock = ptr.Int(queryParameters.Flags.OffersInSideblock())
	}
	var filtersDefaults = ppResponseBuilder.buildFiltersDefaults(searchResult.Filters, queryParameters.Lang)

	primaryPartnerCode := ""
	if queryParameters.PartnerCode != nil {
		primaryPartnerCode = *queryParameters.PartnerCode
	}

	return &responses.PointToPointResponse{
		WizardType: wizardType.String(),
		GreenURL: ppResponseBuilder.buildGreenurl(
			searchResultLanding,
			landingParameters,
			fromSettlement, toSettlement,
			toCrimea,
			queryParameters.Lang,
			queryParameters.Company,
			appliedFilters,
			&landingBuilder,
			queryParameters,
		),
		Context: nil,
		Title: responses.TitlePP{
			URL: ppResponseBuilder.createTitleURL(searchResultLanding, appliedFilters),
			Text: responses.Text{
				Hl: ppResponseBuilder.createTitleText(
					fromSettlement, toSettlement,
					fromSettlementTitle, toSettlementTitle,
					toCrimea,
					queryParameters.Lang,
					queryParameters.Company,
				),
			},
			TitleDescription: ppResponseBuilder.getTitleDescription(
				fromSettlementTitle,
				toSettlementTitle,
				queryParameters.Lang,
				toCrimea,
			),
		},
		Button: responses.URL{
			URL:  routesLanding.WithParam("utm_content", "more_offers").AsString(),
			Text: ppResponseBuilder.buildButtonText(queryParameters.Lang),
		},
		Factors: ppResponseBuilder.buildFactors(fromSettlement, toSettlement, offers),
		Flags:   queryParameters.Flags,
		Content: responses.Content{
			Partners:              ppResponseBuilder.buildPartners(variants, queryParameters, boyIsEnabled, filtersDefaults),
			Offers:                offers,
			Companies:             ppResponseBuilder.buildCompanies(variants, queryParameters.Lang),
			Version:               &searchResult.Version,
			PartnersPollingStatus: pollingStatus,
		},
		To: responses.Point{
			PointKey: toSettlement.GetPointKey(),
			ID:       toSettlement.ID,
			Title:    toSettlementTitle,
		},
		From: responses.Point{
			PointKey: fromSettlement.GetPointKey(),
			ID:       fromSettlement.ID,
			Title:    fromSettlementTitle,
		},
		Reqid:              queryParameters.MainReqID,
		AppliedFilters:     appliedFilters,
		Params:             ppResponseBuilder.buildParams(queryParameters, landingParameters),
		CounterPrefix:      fmt.Sprintf("snippet/buy_tickets/%s", utmCampaign),
		SearchProps:        nil,
		Type:               handlerType,
		FiltersDefaults:    filtersDefaults,
		PrimaryPartnerCode: primaryPartnerCode,
	}, nil
}

func (ppResponseBuilder *PointToPointResponseBuilder) setReturnDate(
	searchResult *results.SearchResult,
	offers *responses.Offers,
	queryParameters *parameters.QueryParameters,
) {
	if searchResult.DateBackward != nil {
		offers.ReturnDate = ptr.String(searchResult.DateBackward.Format("2006-01-02"))
	} else if !helpers.IsNil(queryParameters.ReturnDate) && helpers.IsNil(offers.ReturnDate) {
		offers.ReturnDate = ptr.String(queryParameters.ReturnDate.Format("2006-01-02"))
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildParams(
	queryParameters *parameters.QueryParameters,
	landingParams map[string]string,
) map[string]interface{} {
	params := make(map[string]interface{})
	for k := range landingParams {
		params[k] = landingParams[k]
	}
	params["passengers"] = responses.Passengers(queryParameters.Passengers())
	return params
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildTitleTariff(
	variants []*responses.Variant,
	prices []models.ForwardDayPrice,
	filterState *results.FilterState,
	currency *models.Currency,
) *responses.TitleTariff {

	if len(variants) > 0 {
		titleTariff := responses.TitleTariff{Price: math.MaxInt32}

		for _, v := range variants {
			if v.Tariff.Price < titleTariff.Price {
				titleTariff.Price = v.Tariff.Price
				titleTariff.Currency = v.Tariff.Currency
			}
		}
		return &titleTariff
	}
	if filterState != nil && filterState.Prices != nil && filterState.Prices.Airports != nil {
		minTariffValue := ptr.Float64(math.MaxInt32)
		var currency string
		for _, v := range filterState.Prices.Airports.From {
			if v.Value < *minTariffValue {
				*minTariffValue = v.Value
				currency = v.Currency
			}
		}
		for _, v := range filterState.Prices.Airports.To {
			if v.Value < *minTariffValue {
				*minTariffValue = v.Value
				currency = v.Currency
			}
		}
		if *minTariffValue != math.MaxInt32 {
			return &responses.TitleTariff{
				Currency: currency,
				Price:    *minTariffValue,
			}
		}
	}

	if len(prices) > 0 {
		minPriceValue := float32(math.MaxFloat32)
		for _, p := range prices {
			if p.Price < minPriceValue {
				minPriceValue = p.Price
			}
		}
		titleTariff := &responses.TitleTariff{Currency: currency.Code}
		if minPriceValue > 10000 {
			titleTariff.Price = float64(int(minPriceValue) / 100 * 100)
		} else if minPriceValue > 1000 {
			titleTariff.Price = float64(int(minPriceValue) / 10 * 10)
		} else {
			titleTariff.Price = float64(minPriceValue)
		}
		return titleTariff
	}
	return nil
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildCompanies(
	variants []*responses.Variant,
	lang models.Lang,
) []*responses.Company {
	companies := []*responses.Company{}
	processedIDs := containers.NewSetOfInt()
	for _, v := range variants {
		for _, f := range append(v.Forward.Flights, v.Backward.Flights...) {
			if !processedIDs.Contains(f.CompanyID) {
				companies = append(companies, ppResponseBuilder.buildCompany(f.CompanyID, lang))
				processedIDs.Add(f.CompanyID)
			}
		}
	}
	return companies
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildCompany(companyID int, lang models.Lang) *responses.Company {
	company, _ := ppResponseBuilder.companyRepository.GetByID(companyID)
	baggageType := ppResponseBuilder.variantsBuilder.getCompanyBaggageCostType(companyID)
	title, _ := ppResponseBuilder.translatedTitleRepository.GetOldTitleTranslation(company, lang)
	return &responses.Company{
		Logo: ppResponseBuilder.companyRepository.GetLogoByID(companyID),
		Baggage: &responses.CompanyBaggage{
			CostType: baggageType.String(),
			Title:    ppResponseBuilder.pointToPointTranslator.LocBaggageInfo(baggageType, lang),
		},
		ID:    company.ID,
		Title: title,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildGreenurl(
	searchResultLanding *landings.Landing,
	landingParameters map[string]string,
	fromSettlement, toSettlement *models.Settlement,
	toCrimea bool,
	lang models.Lang,
	company *models.Company,
	filters *responses.AppliedFilters,
	landingBuilder *landings.FrontLandingBuilder,
	queryParameters *parameters.QueryParameters,
) []responses.URL {
	return []responses.URL{
		{
			URL: landingBuilder.
				Morda(queryParameters.UseTravelPortal()).
				WithParams(landingParameters).
				WithParam("utm_content", "greenurl_1").
				AsString(),
			Text: ppResponseBuilder.pointToPointTranslator.LocServiceName(lang, queryParameters.UseTravelPortal()),
		},
		{
			URL:  ppResponseBuilder.createGreenurl2(searchResultLanding, filters),
			Text: ppResponseBuilder.createGreenurl2Text(fromSettlement, toSettlement, toCrimea, lang, company),
		},
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) createGreenurl2Text(
	fromSettlement, toSettlement *models.Settlement,
	toCrimea bool,
	lang models.Lang,
	company *models.Company,
) string {
	if toCrimea {
		toSettlement = &models.Settlement{NewLTitleID: consts.FakeCrimeaTitleID}
	}
	if company != nil {
		companyTitle, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
			company.NewLTitleID,
			lang,
			consts.CaseNominative,
		)
		return ppResponseBuilder.pointToPointTranslator.LocGreenurl2WithCompany(
			companyTitle,
			fromSettlement,
			toSettlement,
			lang,
		)
	}
	return ppResponseBuilder.pointToPointTranslator.LocGreenurl2(fromSettlement, toSettlement, lang)
}

func (ppResponseBuilder *PointToPointResponseBuilder) createGreenurl2(
	searchResultLanding *landings.Landing,
	filters *responses.AppliedFilters,
) string {
	landing := searchResultLanding.WithParam("utm_content", "greenurl_2")
	return ppResponseBuilder.addLandingAnchors(landing, filters).AsString()
}

func (ppResponseBuilder *PointToPointResponseBuilder) createTitleURL(
	searchResultLanding *landings.Landing,
	filters *responses.AppliedFilters,
) string {
	landing := searchResultLanding.WithParam("utm_content", "title")
	landing = ppResponseBuilder.addLandingAnchors(landing, filters)
	return landing.AsString()
}

func (ppResponseBuilder *PointToPointResponseBuilder) addLandingAnchors(
	landing *landings.Landing,
	filters *responses.AppliedFilters,
) *landings.Landing {
	additionalAnchors := make(map[string]string)
	if filters != nil && filters.Transfer != nil && filters.Transfer.Count != nil && *filters.Transfer.Count == 0 {
		additionalAnchors["tt"] = "1"
	}
	if filters != nil && filters.Airlines != nil {
		airlines := make([]string, 0, len(*filters.Airlines))
		for airline := range *filters.Airlines {
			airlines = append(airlines, strconv.Itoa(airline))
		}
		additionalAnchors["c"] = strings.Join(airlines, ",")
	}
	// TODO: RASPTICKETS-17580 Add airport anchors

	return landing.WithAnchors(additionalAnchors)
}

func (ppResponseBuilder *PointToPointResponseBuilder) createTitleText(
	fromSettlement, toSettlement *models.Settlement,
	fromSettlementTitle, toSettlementTitle string,
	toCrimea bool,
	lang models.Lang,
	company *models.Company,
) string {
	if toCrimea {
		toSettlement = &models.Settlement{NewLTitleID: consts.FakeCrimeaTitleID}
	}
	if company != nil {
		companyTitle, _ := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
			company.NewLTitleID,
			lang,
			consts.CaseNominative,
		)
		return ppResponseBuilder.pointToPointTranslator.LocTitleWithCompany(
			companyTitle,
			fromSettlement,
			toSettlement,
			lang,
		)
	}
	return ppResponseBuilder.pointToPointTranslator.LocTitle(
		fromSettlementTitle, toSettlementTitle,
		toSettlement.TitleRuPrepositionVVoNa,
		lang,
	)
}

func (ppResponseBuilder *PointToPointResponseBuilder) getTitleDescription(
	fromSettlementTitle, toSettlementTitle string,
	lang models.Lang,
	toCrimea bool,
) string {
	if toCrimea {
		toSettlementTitle = ppResponseBuilder.translatedTitleRepository.GetCrimeaTranslation(lang)
	}
	return ppResponseBuilder.pointToPointTranslator.LocTitleDescription(fromSettlementTitle, toSettlementTitle, lang)
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildButtonText(lang models.Lang) string {
	return ppResponseBuilder.commonTranslator.LocButton(lang)
}

func needOverrideRedirectURL(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
) bool {
	return !queryParameters.IsDynamic() &&
		!props.SearchPropsContains(
			ctx,
			"date_from_gar",
		) &&
		queryParameters.IsTouchDevice()
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildAppliedFilters(queryParameters *parameters.QueryParameters) *responses.AppliedFilters {
	filters := queryParameters.Filters()
	transfer, transferOk := filters.Transfer()
	withBaggage, withBaggageOk := filters.WithBaggage()
	airlines, airlinesOk := filters.Airlines()
	partners, partnersOk := filters.Partners()
	if (!transferOk || transfer == nil) && (!withBaggageOk || withBaggage == nil) && (!airlinesOk || airlines == nil) && (!partnersOk || partners == nil) {
		return &responses.AppliedFilters{}
	}
	appliedFilters := &responses.AppliedFilters{}
	appliedFilters.WithBaggage, _ = filters.WithBaggage()
	if transferOk && transfer != nil {
		appliedFilters.Transfer = &responses.TransferFilter{}
		appliedFilters.Transfer.Count, _ = transfer.Count()
	}
	if airlinesOk && airlines != nil {
		appliedFilters.Airlines = &airlines
	}
	if partnersOk && partners != nil {
		appliedFilters.Partners = &partners
	}
	return appliedFilters
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildFactors(
	from, to *models.Settlement,
	offers *responses.Offers,
) responses.Factors {
	factors := responses.Factors{
		Distance:            ppResponseBuilder.nearestAviaSettlementProvider.GetDistanceInKMBetween(from, to),
		OffersReturned:      ptr.Int(len(offers.Variants)),
		OffersTotal:         ptr.Int(offers.Total),
		TypePp:              ptr.Int(1),
		DirectionPopularity: ptr.Int(ppResponseBuilder.directionRepository.GetPopularity(from.ID, to.ID)),
		FlightCount: ptr.Int(
			ppResponseBuilder.routeCountRepository.GetCount(
				from.GetPointKey(),
				to.GetPointKey(),
			),
		),
		VariantsShown: ptr.Int(int(math.Min(float64(offers.Show), float64(len(offers.Variants))))),
	}
	if offers.TitleTariff != nil {
		factors.MinPrice = ptr.Int(int(offers.TitleTariff.Price))
	}
	if len(offers.Variants) == 0 {
		return factors
	}
	minForwardSegments := math.MaxInt32
	minBackwardSegments := math.MaxInt32
	minForwardDuration := math.MaxInt32
	minBackwardDuration := math.MaxInt32
	for _, variant := range offers.Variants {
		if len(variant.Forward.Flights) < minForwardSegments {
			minForwardSegments = len(variant.Forward.Flights)
		}
		if len(variant.Backward.Flights) < minBackwardSegments {
			minBackwardSegments = len(variant.Backward.Flights)
		}
		if variant.Forward.Duration < minForwardDuration {
			minForwardDuration = variant.Forward.Duration
		}
		if variant.Backward.Duration < minBackwardDuration {
			minBackwardDuration = variant.Backward.Duration
		}
	}
	factors.MinForwardSegments = ptr.Int(minForwardSegments)
	factors.MinBackwardSegments = ptr.Int(minBackwardSegments)
	factors.MinForwardDuration = ptr.Int(minForwardDuration)
	factors.MinBackwardDuration = ptr.Int(minBackwardDuration)
	return factors
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildFiltersDefaults(
	filterState *results.FilterState,
	lang models.Lang,
) *responses.FiltersDefaults {
	if filterState == nil {
		return nil
	}

	return &responses.FiltersDefaults{
		Transfer:    (*responses.TransferFilter)(filterState.Transfer),
		Airlines:    ppResponseBuilder.buildAirlinesFilterDefaults(filterState, lang),
		Airport:     ppResponseBuilder.buildAirportFilterDefaults(filterState, lang),
		WithBaggage: filterState.WithBaggage,
		Time:        (*responses.TimeFilter)(filterState.Time),
		Prices:      ppResponseBuilder.buildPricesFilterDefaults(filterState),
		AllAirlines: filterState.AllAirlines,
		Partners:    filterState.Partners,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildAirlinesFilterDefaults(
	filterState *results.FilterState,
	lang models.Lang,
) []*responses.AirlineFilter {
	if filterState.Airlines == nil {
		return nil
	}
	var airlines []*responses.AirlineFilter
	for airlineID := range *filterState.Airlines {
		company, _ := ppResponseBuilder.companyRepository.GetByID(airlineID)
		title, _ := ppResponseBuilder.translatedTitleRepository.GetOldTitleTranslation(company, lang)
		airlines = append(
			airlines, &responses.AirlineFilter{
				Logo:  ppResponseBuilder.companyRepository.GetLogoByID(airlineID),
				ID:    airlineID,
				Title: title,
			},
		)
	}
	return airlines
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildPricesFilterDefaults(
	filterState *results.FilterState,
) *responses.Prices {
	if filterState.Prices == nil {
		return nil
	}
	prices := &responses.Prices{
		DirectAirlines: filterState.Prices.DirectAirlines,
	}
	if !helpers.IsNil(filterState.Prices.Airports) {
		prices.Airports = &responses.AirportsPrices{}
		if len(filterState.Prices.Airports.From) > 0 {
			prices.Airports.From = ppResponseBuilder.convertPrices(filterState.Prices.Airports.From)
		}
		if len(filterState.Prices.Airports.To) > 0 {
			prices.Airports.To = ppResponseBuilder.convertPrices(filterState.Prices.Airports.To)
		}
	}
	if !helpers.IsNil(filterState.Airlines) {
		airlinesPrices := ppResponseBuilder.convertPrices(*filterState.Prices.Airlines)
		prices.Airlines = airlinesPrices
	}
	if !helpers.IsNil(filterState.Prices.Transfers) {
		prices.Transfers = ppResponseBuilder.convertTransfersPrices(*filterState.Prices.Transfers)
	}
	if !helpers.IsNil(filterState.Prices.WithBaggage) {
		prices.WithBaggage = &responses.PriceValue{
			Currency: filterState.Prices.WithBaggage.Currency,
			Value:    filterState.Prices.WithBaggage.Value,
		}
	}
	if !helpers.IsNil(filterState.Prices.DirectFlight) {
		prices.DirectFlight = &responses.PriceValue{
			Currency: filterState.Prices.DirectFlight.Currency,
			Value:    filterState.Prices.DirectFlight.Value,
		}
	}
	return prices
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildAirportFilterDefaults(
	filterState *results.FilterState,
	lang models.Lang,
) *responses.AirportFilter {
	if filterState.Airport == nil {
		return nil
	}
	return &responses.AirportFilter{
		ForwardDeparture:  ppResponseBuilder.buildAirportFilterValues(filterState.Airport.ForwardDeparture, lang),
		ForwardTransfers:  ppResponseBuilder.buildAirportFilterValues(filterState.Airport.ForwardTransfers, lang),
		ForwardArrival:    ppResponseBuilder.buildAirportFilterValues(filterState.Airport.ForwardArrival, lang),
		BackwardDeparture: ppResponseBuilder.buildAirportFilterValues(filterState.Airport.BackwardDeparture, lang),
		BackwardTransfers: ppResponseBuilder.buildAirportFilterValues(filterState.Airport.BackwardTransfers, lang),
		BackwardArrival:   ppResponseBuilder.buildAirportFilterValues(filterState.Airport.BackwardArrival, lang),
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildAirportFilterValues(
	airportIDs containers.SetOfInt,
	lang models.Lang,
) []*responses.AirportFilterValue {
	airportFilterValues := make([]*responses.AirportFilterValue, 0)
	for airportID := range airportIDs {
		if filterValue := ppResponseBuilder.buildAirportFilterValue(airportID, lang); filterValue != nil {
			airportFilterValues = append(airportFilterValues, filterValue)
		}
	}
	return airportFilterValues
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildAirportFilterValue(
	airportID int,
	lang models.Lang,
) *responses.AirportFilterValue {
	station, found := ppResponseBuilder.stationRepository.GetByID(airportID)
	if !found {
		return nil
	}
	title, err := ppResponseBuilder.translatedTitleRepository.GetTitleTranslation(
		station.NewLTitleID,
		lang,
		consts.CaseNominative,
	)
	if err != nil {
		return nil
	}
	return &responses.AirportFilterValue{
		ID:    airportID,
		Title: title,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) convertPrices(prices map[string]*results.Price) map[string]*responses.Price {
	newPrices := make(map[string]*responses.Price)
	for id, price := range prices {
		if !helpers.IsDigit(id) {
			continue
		}
		newPrices[id] = convertPrice(price)
	}
	return newPrices
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildPartners(
	variants []*responses.Variant, queryParameters *parameters.QueryParameters, boyIsEnabled bool, filtersDefaults *responses.FiltersDefaults,
) []*responses.Partner {
	partners := []*responses.Partner{}
	partnerCodes := make(containers.SetOfString)
	for _, v := range variants {
		if partnerCodes.Contains(v.PartnerCode) {
			continue
		}
		partners = append(partners, ppResponseBuilder.buildPartner(v.PartnerCode, queryParameters, boyIsEnabled))
		partnerCodes.Add(v.PartnerCode)
	}
	if filtersDefaults != nil && filtersDefaults.Partners != nil {
		for partnerCode := range *filtersDefaults.Partners {
			if partnerCodes.Contains(partnerCode) {
				continue
			}
			partners = append(partners, ppResponseBuilder.buildPartner(partnerCode, queryParameters, boyIsEnabled))
			partnerCodes.Add(partnerCode)
		}
	}
	return partners
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildPartner(
	partnerCode string,
	queryParameters *parameters.QueryParameters,
	boyIsEnabled bool,
) *responses.Partner {
	if partnerCode == "ticketsru" && queryParameters.Tld == "kz" {
		return &responses.Partner{
			Code:  "ticketsru",
			Title: "tickets.kz",
			URL:   "tickets.kz",
			Icon:  "",
		}
	}

	if partnerCode == "aeroflot" && boyIsEnabled {
		return &responses.Partner{
			Code:  "aeroflot",
			Title: "travel.yandex.ru",
			URL:   "travel.yandex.ru",
			Icon:  "",
		}
	}

	url := ""
	icon := ""
	title := ""
	partner, _ := ppResponseBuilder.partnerRepository.GetPartnerByCode(partnerCode)
	if partner != nil {
		url = partner.SiteURL
		icon = partner.IconSvg

		nv := queryParameters.NationalVersion
		if partner.Titles != nil {
			title = partner.Titles[nv]
		}
	}
	if url == "" {
		url, _ = ppResponseBuilder.translatedTitleRepository.GetOldTitleTranslation(partner, queryParameters.Lang)
	}
	if url == "" {
		ppResponseBuilder.logger.Warn(
			fmt.Sprintf("can not get url for partner %+v with partner code %s", partner, partnerCode),
			log.String("job_id", queryParameters.JobID),
		)
	}
	if title == "" {
		title = url
	}
	return &responses.Partner{
		Code:  partnerCode,
		Title: title,
		URL:   url,
		Icon:  icon,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) convertTransfersPrices(prices map[string]*results.Price) map[string]*responses.PriceValue {
	newPrices := make(map[string]*responses.PriceValue)
	for key := range prices {
		if helpers.IsNil(prices[key]) {
			newPrices[key] = nil
			continue
		}
		newPrices[key] = &responses.PriceValue{
			Currency: prices[key].Currency,
			Value:    prices[key].Value,
		}
	}
	return newPrices
}

func (ppResponseBuilder *PointToPointResponseBuilder) setMinDateMaxDate(
	routesLanding *landings.Landing,
	variants []*responses.Variant,
	prices []models.ForwardDayPrice,
) *landings.Landing {
	if len(variants) > 0 {
		minMaxDate := variants[0].Forward.Flights[0].DepartsAt[:len("YYYY-mm-dd")]
		return routesLanding.WithParams(
			map[string]string{
				"minDate": minMaxDate,
				"maxDate": minMaxDate,
			},
		)
	}
	if len(prices) > 0 {
		minDate := prices[0].DateForward
		maxDate := prices[0].DateForward
		for _, p := range prices {
			if p.DateForward < minDate {
				minDate = p.DateForward
			}
			if p.DateForward > maxDate {
				maxDate = p.DateForward
			}
		}
		return routesLanding.WithParams(
			map[string]string{
				"minDate": minDate,
				"maxDate": maxDate,
			},
		)
	}
	return routesLanding
}

func (ppResponseBuilder *PointToPointResponseBuilder) buildPollingStatus(
	pollingStatus *results.PollingStatus,
	queryParameters *parameters.QueryParameters,
	boyIsEnabled bool,
) *responses.PollingStatus {
	if pollingStatus == nil || !queryParameters.IsDynamic() {
		return nil
	}
	askedPartners := []*responses.Partner{}
	for _, partnerCode := range pollingStatus.AskedPartners {
		askedPartners = append(askedPartners, ppResponseBuilder.buildPartner(partnerCode, queryParameters, boyIsEnabled))
	}
	remainingPartners := []*responses.Partner{}
	for _, partnerCode := range pollingStatus.RemainingPartners {
		remainingPartners = append(remainingPartners, ppResponseBuilder.buildPartner(partnerCode, queryParameters, boyIsEnabled))
	}
	return &responses.PollingStatus{
		AskedPartners:          askedPartners,
		AskedPartnersCount:     pollingStatus.AskedPartnersCount,
		RemainingPartners:      remainingPartners,
		RemainingPartnersCount: pollingStatus.RemainingPartnersCount,
	}
}

func (ppResponseBuilder *PointToPointResponseBuilder) hasAnyAppliedFilters(appliedFilters *responses.AppliedFilters) bool {
	if appliedFilters.Airlines != nil && len(*appliedFilters.Airlines) > 0 {
		return true
	}
	if appliedFilters.Partners != nil && len(*appliedFilters.Partners) > 0 {
		return true
	}
	if appliedFilters.WithBaggage != nil && *appliedFilters.WithBaggage {
		return true
	}
	transfer := appliedFilters.Transfer
	return transfer != nil && (transfer.Count != nil ||
		transfer.HasNight != nil && *transfer.HasNight ||
		transfer.HasAirportChange != nil && !*transfer.HasAirportChange ||
		transfer.MinDuration != nil || transfer.MaxDuration != nil)
	// TODO: RASPTICKETS-17580 Check Airports filter
	// if filters.Airport != nil && (
	// 	filters.Airport.ForwardDeparutre != nil && len(*filters.Airport.ForwardDeparutre) > 0 ||
	// 		filters.Airport.ForwardTransfers != nil && len(*filters.Airport.ForwardTransfers) > 0 ||
	// 		filters.Airport.ForwardArrival != nil && len(*filters.Airport.ForwardArrival) > 0 ||
	// 		filters.Airport.BackwardDeparutre != nil && len(*filters.Airport.BackwardDeparutre) > 0 ||
	// 		filters.Airport.BackwardTransfers != nil && len(*filters.Airport.BackwardTransfers) > 0 ||
	// 		filters.Airport.BackwardArrival != nil && len(*filters.Airport.BackwardArrival) > 0) {
	// 	return true
	// }
}

func (ppResponseBuilder *PointToPointResponseBuilder) writeStats(
	queryParameters *parameters.QueryParameters,
	isCacheMiss, noVariants bool,
) {
	if noVariants {
		ppResponseBuilder.logger.Info(
			fmt.Sprintf("YAK for main_reqid: %s reqid: %s", queryParameters.MainReqID, queryParameters.ReqID),
			log.String("job_id", queryParameters.JobID),
		)
	}
	if !isCacheMiss && noVariants {
		metrics.GlobalWizardMetrics().GetYakCounter(queryParameters.IsDynamic()).Inc()
		metrics.GlobalWizardMetrics().GetYakCounterByNationalVersion(
			queryParameters.NationalVersion,
			queryParameters.IsDynamic(),
		).Inc()
	}

	metrics.GlobalWizardMetrics().GetPPResponseTypeCounter(queryParameters.IsDynamic(), noVariants).Inc()
	metrics.GlobalWizardMetrics().GetPPResponseTypeByNationalVersionCounter(
		queryParameters.NationalVersion,
		queryParameters.IsDynamic(),
		noVariants,
	).Inc()
}

func convertPrice(price *results.Price) *responses.Price {
	if helpers.IsNil(price) {
		return nil
	}
	return &responses.Price{
		Currency: price.Currency,
		Price:    price.Value,
	}
}
