package building

import (
	"context"
	"fmt"
	"math"
	"sort"
	"strconv"
	"strings"
	"time"

	"github.com/opentracing/opentracing-go"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/show"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type VariantsBuilder struct {
	settlementRepository      repositories.Settlement
	translatedTitleRepository repositories.TranslatedTitle
	pointToPointTranslator    *translations.PointToPointTranslator
	stationRepository         repositories.Station
	aviaCompanyRepository     repositories.AviaCompany
	companyTariffRepository   repositories.CompanyTariff
	urlBuilder                *URLBuilder
	logger                    log.Logger
	showLogger                *show.Logger
}

func NewVariantsBuilder(
	settlementRepository repositories.Settlement,
	translatedTitleRepository repositories.TranslatedTitle,
	pointToPointTranslator *translations.PointToPointTranslator,
	stationRepository repositories.Station,
	aviaCompanyRepository repositories.AviaCompany,
	companyTariffRepository repositories.CompanyTariff,
	urlBuilder *URLBuilder,
	logger log.Logger,
	showLogger *show.Logger,
) *VariantsBuilder {
	return &VariantsBuilder{
		settlementRepository:      settlementRepository,
		translatedTitleRepository: translatedTitleRepository,
		pointToPointTranslator:    pointToPointTranslator,
		stationRepository:         stationRepository,
		aviaCompanyRepository:     aviaCompanyRepository,
		companyTariffRepository:   companyTariffRepository,
		urlBuilder:                urlBuilder,
		logger:                    logger,
		showLogger:                showLogger,
	}
}

func (builder *VariantsBuilder) buildVariants(
	ctx context.Context,
	fareGenerator *results.FareGenerator,
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	landingParameters map[string]string,
	searchResultLanding *landings.Landing,
	landingBuilder *landings.FrontLandingBuilder,
	searchDepth int,
) []*responses.Variant {
	if helpers.IsUkraine(fromSettlement, toSettlement) {
		return []*responses.Variant{}
	}
	now := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().VariantsBuildingTimer.RecordDuration(time.Since(now))
		builder.logger.Info(
			fmt.Sprintf("building variants: %d", time.Since(now).Milliseconds()),
			log.String("job_id", queryParameters.JobID),
		)
	}()
	span, ctx := opentracing.StartSpanFromContext(ctx, "Building variants")
	defer span.Finish()
	variants := []*responses.Variant{}
	variantsLimit := queryParameters.Context().OffersToShow() + queryParameters.Context().OffersUnderCut()
	for {
		fare, err := fareGenerator.Next(ctx)
		if err != nil {
			break
		}
		variant, err := builder.buildVariant(fare, queryParameters.Lang)
		if err != nil {
			continue
		}
		variants = append(variants, variant)
		if !queryParameters.Flags.SortByPrice() && len(variants) == variantsLimit {
			break
		}
	}
	variants = builder.getSortedVariants(variants, queryParameters)
	variants = variants[:int(math.Min(float64(variantsLimit), float64(len(variants))))]
	for i, variant := range variants {
		showID := fmt.Sprintf("%s|%d", queryParameters.JobID, i)
		variant.ShowID = showID
		builder.setVariantURLs(
			ctx,
			variant,
			fromSettlement, toSettlement,
			landingParameters,
			queryParameters,
			searchResultLanding,
			landingBuilder,
			searchDepth,
		)
		go func(v *responses.Variant, index int) {
			builder.showLogger.Log(show.NewRecord(queryParameters, v, index, v.BaggageType))
		}(variant, i)
	}
	return variants
}

func (builder *VariantsBuilder) buildVariant(fare *results.Fare, lang models.Lang) (
	*responses.Variant,
	error,
) {
	errorGroup := errgroup.Group{}
	var forward *responses.Trip
	var backward *responses.Trip
	errorGroup.Go(
		func() error {
			var err error
			forward, err = builder.buildTrip(fare.ForwardTrip, lang)
			return err
		},
	)
	errorGroup.Go(
		func() error {
			var err error
			backward, err = builder.buildTrip(fare.BackwardTrip, lang)
			return err
		},
	)
	var variantBaggageType models.BaggageType
	errorGroup.Go(
		func() error {
			variantBaggageType = builder.getVariantBaggageType(fare)
			return nil
		},
	)

	var variantType *responses.VariantType
	if fare.IsPromo && lang == consts.LangRU {
		variantType = builder.buildVariantType(models.VariantTypePromo, lang)
	}

	if err := errorGroup.Wait(); err != nil {
		return nil, err
	}
	return &responses.Variant{
		ShowID:   "", // Will be set later
		OrderURL: "", // Will be set later
		URL:      "", // Will be set later
		Snippet:  "", // Will be set later
		Button:   "", // Will be set later

		PartnerCode: fare.PartnerCode,
		CreatedAt:   fare.CreatedAt,
		Qid:         fare.Qid,
		Tariff: responses.Tariff{
			Currency: fare.Tariff.Currency,
			Price:    fare.Tariff.Value,
		},
		Popularity:           fare.Popularity,
		BaggageInfo:          builder.pointToPointTranslator.LocBaggageInfo(variantBaggageType, lang),
		Baggage:              builder.buildVariantBaggage(fare),
		Forward:              forward,
		Backward:             backward,
		OfferFromAviacompany: fare.FromAviaCompany,
		BaggageType:          variantBaggageType,
		Type:                 variantType,
	}, nil
}

func (builder *VariantsBuilder) getVariantBaggageType(fare *results.Fare) models.BaggageType {
	baggageTypes := make(containers.SetOfString)
	for _, flight := range fare.ForwardTrip {
		baggageTypes.Add(builder.getCompanyBaggageCostType(flight.CompanyID).String())
	}
	for _, flight := range fare.BackwardTrip {
		baggageTypes.Add(builder.getCompanyBaggageCostType(flight.CompanyID).String())
	}
	if baggageTypes.Contains(models.BaggageTypeLowcost.String()) {
		return models.BaggageTypeLowcost
	}
	if len(baggageTypes) == 1 && baggageTypes.Contains(models.BaggageTypeNormal.String()) {
		return models.BaggageTypeNormal
	}
	return models.BaggageTypeHybrid
}

func (builder *VariantsBuilder) getCompanyBaggageCostType(companyID int) models.BaggageType {
	aviaCompany, found := builder.aviaCompanyRepository.GetByID(companyID)
	if !found {
		return models.BaggageTypeHybrid
	}
	companyTariffs, found := builder.companyTariffRepository.GetTariffsByCompanyID(aviaCompany.RaspCompanyID)
	if found {
		for _, t := range companyTariffs {
			if t.BaggageAllowed == 0 {
				return models.BaggageType(aviaCompany.CostType)
			}
		}
	}
	return models.BaggageTypeNormal
}

func (builder *VariantsBuilder) buildTrip(trip results.Trip, lang models.Lang) (*responses.Trip, error) {
	flights := make([]*responses.PPFlight, 0, len(trip))
	for i, segment := range trip {
		arrival, err := builder.buildStation(segment.StationTo, lang)
		if err != nil {
			return nil, err
		}
		departure, err := builder.buildStation(segment.StationFrom, lang)
		if err != nil {
			return nil, err
		}

		flights = append(
			flights, &responses.PPFlight{
				Arrival:   arrival,
				Departure: departure,
				Number:    segment.Number,
				CompanyID: segment.CompanyID,
				ArrivesAt: segment.Arrival.Format("2006-01-02T15:04:05"),
				DepartsAt: segment.Departure.Format("2006-01-02T15:04:05"),
			},
		)
		if i > 0 {
			previousSegment := trip[i-1]
			flights[i].Stopover = &responses.Stopover{
				Duration:      int(segment.Departure.Sub(previousSegment.Arrival).Seconds()),
				AirportChange: previousSegment.StationTo.ID != segment.StationFrom.ID,
				Overnight:     !helpers.TruncateToDate(previousSegment.Arrival).Equal(helpers.TruncateToDate(segment.Departure)),
			}
		}
	}

	return &responses.Trip{
		Duration:    int(trip.Duration().Seconds()),
		Flights:     flights,
		RedirectKey: trip.BuildRedirectKey(),
	}, nil
}

func (builder *VariantsBuilder) buildStation(station *models.Station, lang models.Lang) (
	*responses.RoutePoint,
	error,
) {
	settlement, found := builder.settlementRepository.GetByID(station.SettlementID)
	if !found {
		return nil, domain.NewWizardError(fmt.Sprintf("unknown settlement: %d", station.SettlementID), domain.BadSettlement)
	}
	settlementTitle, err := builder.translatedTitleRepository.GetTitleTranslation(
		settlement.NewLTitleID,
		lang,
		consts.CaseNominative,
	)
	if err != nil {
		return nil, domain.NewWizardError(
			fmt.Sprintf("no title translation for settlement with id: %d", settlement.ID),
			domain.NoTranslation,
		)
	}
	stationTitle, err := builder.translatedTitleRepository.GetTitleTranslation(
		station.NewLTitleID,
		lang,
		consts.CaseNominative,
	)
	if err != nil {
		return nil, domain.NewWizardError(
			fmt.Sprintf("no title translation for station with id: %d", station.ID),
			domain.NoTranslation,
		)
	}
	settlementCode, found := builder.settlementRepository.GetAnyCodeByID(station.SettlementID)
	if !found {
		return nil, domain.NewWizardError(
			fmt.Sprintf("no code for settlement with id %d", station.SettlementID),
			domain.NoCode,
		)
	}
	stationCode, found := builder.stationRepository.GetAnyCodeByID(station.ID)
	if !found {
		return nil, domain.NewWizardError(
			fmt.Sprintf("no code for station with id %d", station.SettlementID),
			domain.NoCode,
		)
	}
	settlementTitleLocativeWithPreposition := builder.translatedTitleRepository.LocTitleLocativeWithPreposition(
		settlement.NewLTitleID,
		settlement.TitleRuPrepositionVVoNa,
		lang,
	)
	stationTitleLocativeWithPreposition := builder.translatedTitleRepository.LocTitleLocativeWithPreposition(
		station.NewLTitleID,
		station.TitleRuPrepositionVVoNa,
		lang,
	)
	stationIATACode, _ := builder.stationRepository.GetIATACodeByID(station.ID)
	return &responses.RoutePoint{
		Settlement: responses.Settlement{
			Code:          settlementCode,
			TitleLocative: settlementTitleLocativeWithPreposition,
			Title:         settlementTitle,
		},
		Station: responses.Station{
			Code:          stationCode,
			TitleLocative: stationTitleLocativeWithPreposition,
			Title:         stationTitle,
			IATA:          stationIATACode,
			PointKey:      station.GetPointKey(),
		},
	}, nil
}

func (builder *VariantsBuilder) getSortedVariants(
	variants []*responses.Variant,
	queryParameters *parameters.QueryParameters,
) []*responses.Variant {
	if len(variants) <= 1 {
		return variants
	}

	if queryParameters.Flags.SortByPrice() {
		return builder.sortVariantsByPrice(variants, queryParameters)
	}
	return builder.getVariantsSortedByDefault(variants, queryParameters)
}

func (builder *VariantsBuilder) sortVariantsByPrice(
	variants []*responses.Variant,
	queryParameters *parameters.QueryParameters,
) []*responses.Variant {
	buildKey := func(v *responses.Variant) string {
		forwardCompanies := make([]string, 0, len(v.Forward.Flights))
		for _, flight := range v.Forward.Flights {
			forwardCompanies = append(forwardCompanies, strconv.Itoa(flight.CompanyID))
		}
		sort.SliceStable(
			forwardCompanies, func(i, j int) bool {
				return forwardCompanies[i] < forwardCompanies[j]
			},
		)
		backwardCompanies := make([]string, 0, len(v.Backward.Flights))
		for _, flight := range v.Backward.Flights {
			backwardCompanies = append(backwardCompanies, strconv.Itoa(flight.CompanyID))
		}
		sort.SliceStable(
			backwardCompanies, func(i, j int) bool {
				return backwardCompanies[i] < backwardCompanies[j]
			},
		)
		return fmt.Sprintf(
			"%s;%s;%s;%s",
			v.PartnerCode,
			strings.Join(forwardCompanies, ","),
			strings.Join(backwardCompanies, ","),
			fmt.Sprintf("%f", v.Tariff.Price),
		)
	}
	variantByKey := make(map[string]*responses.Variant)
	for _, v := range variants {
		key := buildKey(v)
		if variant, ok := variantByKey[key]; ok && variant.Popularity < v.Popularity || !ok {
			variantByKey[key] = v
		}
	}
	uniqueVariants := []*responses.Variant{}
	for _, variant := range variantByKey {
		uniqueVariants = append(uniqueVariants, variant)
	}
	sort.SliceStable(
		uniqueVariants, func(i, j int) bool {
			return uniqueVariants[i].Tariff.Price < uniqueVariants[j].Tariff.Price
		},
	)
	builder.updateVariantType(uniqueVariants[0], models.VariantTypeMinPrice, queryParameters.Lang)
	return uniqueVariants
}

func (builder *VariantsBuilder) getVariantsSortedByDefault(
	variants []*responses.Variant,
	queryParameters *parameters.QueryParameters,
) []*responses.Variant {
	cheapestVariantIdx := 0
	minPrice := math.MaxFloat64
	for i := 0; i < len(variants); i++ {
		if variants[i].Tariff.Price < minPrice {
			minPrice = variants[i].Tariff.Price
			cheapestVariantIdx = i
		}
	}
	indicesToExclude := containers.NewSetOfInt(cheapestVariantIdx)
	mostPopularVariantIdx := -1
	maxPopularity := int32(math.MinInt32)
	for i := 0; i < len(variants); i++ {
		if !indicesToExclude.Contains(i) && variants[i].Popularity > maxPopularity {
			maxPopularity = variants[i].Popularity
			mostPopularVariantIdx = i
		}
	}
	indicesToExclude.Add(mostPopularVariantIdx)

	builder.updateVariantType(variants[cheapestVariantIdx], models.VariantTypeMinPrice, queryParameters.Lang)
	if mostPopularVariantIdx >= 0 {
		builder.updateVariantType(
			variants[mostPopularVariantIdx],
			models.VariantTypeTopFlight,
			queryParameters.Lang,
		)
	}

	sortedVariants := []*responses.Variant{variants[cheapestVariantIdx]}
	if mostPopularVariantIdx >= 0 {
		sortedVariants = append(sortedVariants, variants[mostPopularVariantIdx])
	}
	for i := 0; i < len(variants); i++ {
		if !indicesToExclude.Contains(i) {
			sortedVariants = append(sortedVariants, variants[i])
		}
	}
	return sortedVariants
}

func (builder *VariantsBuilder) buildVariantBaggage(fare *results.Fare) *responses.Baggage {
	if fare.Baggage == nil {
		return nil
	}
	return &responses.Baggage{
		CarryOn: (*responses.CarryOn)(fare.Baggage.CarryOn),
		Checked: responses.Checked{
			Included:   fare.Baggage.Checked.Included,
			Dimensions: (*responses.Dimensions)(fare.Baggage.Checked.Dimensions),
			Weight:     fare.Baggage.Checked.Weight,
			Pieces:     fare.Baggage.Checked.Pieces,
		},
	}
}

func (builder *VariantsBuilder) updateVariantType(
	variant *responses.Variant,
	variantType models.VariantType,
	lang models.Lang,
) {
	if variant.Type == nil {
		variant.Type = builder.buildVariantType(variantType, lang)
	}

}

func (builder *VariantsBuilder) buildVariantType(
	variantType models.VariantType,
	lang models.Lang,
) *responses.VariantType {
	variantTypeText := builder.pointToPointTranslator.LocVariantType(variantType, lang)
	return &responses.VariantType{
		TextFeatures: []*string{variantTypeText},
		Text:         variantTypeText,
		Code:         variantType.String(),
		Codes:        []string{variantType.String()},
	}
}

func (builder *VariantsBuilder) setVariantURLs(
	ctx context.Context,
	variant *responses.Variant,
	from, to *models.Settlement,
	landingParameters map[string]string,
	queryParameters *parameters.QueryParameters,
	searchResultLanding *landings.Landing,
	landingBuilder *landings.FrontLandingBuilder,
	searchDepth int,
) {
	passengers := queryParameters.Passengers()

	var offerParams map[string]string
	if queryParameters.Flags.UseUnisearch() {
		offerParams = map[string]string{
			"unisearchRedirKey": variant.ShowID,
			"utm_content":       "offer",
			"adult_seats":       strconv.Itoa(passengers.Adults),
			"children_seats":    strconv.Itoa(passengers.Children),
			"infant_seats":      strconv.Itoa(passengers.Infants),
		}
	} else {
		offerParams = map[string]string{
			"wizardRedirKey": variant.ShowID,
			"utm_content":    "offer",
			"adult_seats":    strconv.Itoa(passengers.Adults),
			"children_seats": strconv.Itoa(passengers.Children),
			"infant_seats":   strconv.Itoa(passengers.Infants),
		}
	}

	variant.URL = builder.urlBuilder.createRedirectURL(
		variant,
		queryParameters,
		landingParameters,
		offerParams,
		landingBuilder,
	)

	if queryParameters.Flags.AntiFASOfferRedirects() {
		if buttonRedirectToSearchResultPage(ctx, queryParameters) {
			variant.Button = searchResultLanding.AsString()
		} else if buttonRedirectToOrderPage(ctx, queryParameters, searchDepth) {
			variant.Button = builder.urlBuilder.createOrderURL(
				ctx,
				variant,
				queryParameters,
				landingParameters,
				offerParams,
				searchResultLanding,
				from, to,
				landingBuilder,
			)
		} else {
			variant.Button = variant.URL
		}
	} else {
		variant.Button = variant.URL
	}

	variant.OrderURL = builder.urlBuilder.createOrderURL(
		ctx,
		variant,
		queryParameters,
		landingParameters,
		offerParams,
		searchResultLanding,
		from, to,
		landingBuilder,
	)

	if queryParameters.Flags.AntiFASOfferRedirects() {
		if snippetRedirectToSearchResultPage(ctx, queryParameters) {
			variant.Snippet = searchResultLanding.AsString()
		} else if queryParameters.IsTouchDevice() {
			variant.Snippet = variant.URL
		} else {
			variant.Snippet = variant.OrderURL
		}
	} else {
		variant.Snippet = variant.URL
	}
}
