package building

import (
	"context"
	"fmt"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type VariantsInfoResponseBuilder struct {
	variantsBuilder           *VariantsBuilder
	translatedTitleRepository repositories.TranslatedTitle
	companyRepository         repositories.Company
	translator                *translations.PointToPointTranslator
	landingBuilder            landings.FrontLandingBuilder
	directionRepository       repositories.Direction
	logger                    log.Logger
}

func NewVariantsInfoResponseBuilder(
	variantsBuilder *VariantsBuilder,
	translatedTitleRepository repositories.TranslatedTitle,
	landingBuilder landings.FrontLandingBuilder,
	directionRepository repositories.Direction,
	companyRepository repositories.Company,
	translator *translations.PointToPointTranslator,
	logger log.Logger,
) *VariantsInfoResponseBuilder {
	return &VariantsInfoResponseBuilder{
		variantsBuilder:           variantsBuilder,
		translatedTitleRepository: translatedTitleRepository,
		landingBuilder:            landingBuilder,
		directionRepository:       directionRepository,
		companyRepository:         companyRepository,
		translator:                translator,
		logger:                    logger,
	}
}

func (builder *VariantsInfoResponseBuilder) Build(
	ctx context.Context,
	searchResult *results.SearchResult,
	queryParameters *parameters.QueryParameters,
	landingParameters map[string]string,
	fromSettlement, toSettlement *models.Settlement,
	toCrimea bool,
) (*responses.VariantsInfoResponse, error) {
	span, ctx := opentracing.StartSpanFromContext(ctx, "Building ad feed result")
	defer span.Finish()
	landingBuilder := builder.landingBuilder.WithQueryParameters(queryParameters)

	fromName, _ := builder.translatedTitleRepository.GetTitleTranslation(
		fromSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	toName, _ := builder.translatedTitleRepository.GetTitleTranslation(
		toSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	passengers := queryParameters.Passengers()
	searchResultLanding := landingBuilder.SearchResult(
		fromSettlement, toSettlement,
		fromName, toName,
		searchResult.DateForward, searchResult.DateBackward,
		&passengers,
	).WithParams(landingParameters)
	searchDepth := int(searchResult.DateForward.Sub(helpers.TruncateToDate(time.Now())).Seconds()) / consts.SecondsInDay
	variants := builder.variantsBuilder.buildVariants(
		ctx,
		searchResult.FareGenerator,
		queryParameters,
		fromSettlement,
		toSettlement,
		landingParameters,
		searchResultLanding,
		&landingBuilder,
		searchDepth,
	)

	fromSettlementTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(
		fromSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	toSettlementTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(
		toSettlement.NewLTitleID,
		queryParameters.Lang,
		consts.CaseNominative,
	)
	if toCrimea {
		toSettlementTitle = builder.translatedTitleRepository.GetCrimeaTranslation(queryParameters.Lang)
	}
	noVariants := len(variants) == 0
	nv := queryParameters.NationalVersion
	if nv == "by" {
		nv = "ru"
	}
	if noVariants && !builder.directionRepository.IsPossibleVariant(fromSettlement.ID, toSettlement.ID, nv) {
		builder.logger.Info(
			fmt.Sprintf(
				"no prices for main_reqid: %s, reqid: %s", queryParameters.MainReqID,
				queryParameters.ReqID,
			), log.String("job_id", queryParameters.JobID),
		)
		return nil, domain.NewWizardError("No prices found", domain.NoContent)
	}

	return &responses.VariantsInfoResponse{
		To: responses.Point{
			PointKey: toSettlement.GetPointKey(),
			ID:       toSettlement.ID,
			Title:    toSettlementTitle,
		},
		From: responses.Point{
			PointKey: fromSettlement.GetPointKey(),
			ID:       fromSettlement.ID,
			Title:    fromSettlementTitle,
		},
		Variants:  variants,
		Companies: builder.buildCompanies(variants, queryParameters.Lang),
	}, nil
}

func (builder *VariantsInfoResponseBuilder) buildVariantBaggage(fare *results.Fare) *responses.Baggage {
	if fare.Baggage == nil {
		return nil
	}
	return &responses.Baggage{
		CarryOn: (*responses.CarryOn)(fare.Baggage.CarryOn),
		Checked: responses.Checked{
			Included:   fare.Baggage.Checked.Included,
			Dimensions: (*responses.Dimensions)(fare.Baggage.Checked.Dimensions),
			Weight:     fare.Baggage.Checked.Weight,
			Pieces:     fare.Baggage.Checked.Pieces,
		},
	}
}

func (builder *VariantsInfoResponseBuilder) convertPrices(prices map[string]*results.Price) map[string]*responses.Price {
	newPrices := make(map[string]*responses.Price)
	for id, price := range prices {
		if !helpers.IsDigit(id) {
			continue
		}
		newPrices[id] = convertPrice(price)
	}
	return newPrices
}

func (builder *VariantsInfoResponseBuilder) convertTransfersPrices(prices map[string]*results.Price) map[string]*responses.PriceValue {
	newPrices := make(map[string]*responses.PriceValue)
	for key := range prices {
		if helpers.IsNil(prices[key]) {
			newPrices[key] = nil
			continue
		}
		newPrices[key] = &responses.PriceValue{
			Currency: prices[key].Currency,
			Value:    prices[key].Value,
		}
	}
	return newPrices
}

func (builder *VariantsInfoResponseBuilder) buildCompanies(
	variants []*responses.Variant,
	lang models.Lang,
) []*responses.Company {
	companies := []*responses.Company{}
	for _, v := range variants {
		for _, f := range v.Forward.Flights {
			companies = append(companies, builder.buildCompany(f.CompanyID, lang))
		}
	}
	for _, v := range variants {
		for _, f := range v.Backward.Flights {
			companies = append(companies, builder.buildCompany(f.CompanyID, lang))
		}
	}
	return companies
}

func (builder *VariantsInfoResponseBuilder) buildCompany(companyID int, lang models.Lang) *responses.Company {
	company, _ := builder.companyRepository.GetByID(companyID)
	baggageType := builder.variantsBuilder.getCompanyBaggageCostType(companyID)
	title, _ := builder.translatedTitleRepository.GetOldTitleTranslation(company, lang)
	return &responses.Company{
		Logo: builder.companyRepository.GetLogoByID(companyID),
		Baggage: &responses.CompanyBaggage{
			CostType: baggageType.String(),
			Title:    builder.translator.LocBaggageInfo(baggageType, lang),
		},
		ID:    company.ID,
		Title: title,
	}
}
