package handlers

import (
	"context"
	"time"

	timeformats "cuelang.org/go/pkg/time"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type CommonHandler struct {
	logger               log.Logger
	settlementRepository repositories.Settlement
	pointParser          point.IParser
	regionRepository     repositories.Region
	responseBuilder      *building.CommonResponseBuilder
	commonTranslator     *translations.CommonTranslator
	locationRepository   repositories.CachedLocation
}

func NewCommonHandler(
	logger log.Logger,
	responseBuilder *building.CommonResponseBuilder,
	locationRepository repositories.CachedLocation,
	settlementRepository repositories.Settlement,
	regionRepository repositories.Region,
) *CommonHandler {
	return &CommonHandler{
		logger:               logger,
		responseBuilder:      responseBuilder,
		locationRepository:   locationRepository,
		settlementRepository: settlementRepository,
		regionRepository:     regionRepository,
	}
}

func (handler *CommonHandler) Handle(
	queryParameters *parameters.QueryParameters,
	fromSettlement *models.Settlement,
	landingParameters map[string]string,
	ctx context.Context,
) (responses.HandlerResponse, error) {
	now := time.Now()
	defer func() { metrics.GlobalWizardMetrics().GetCommonRequestTimer().RecordDuration(time.Since(now)) }()

	handlerSpan, _ := opentracing.StartSpanFromContext(ctx, "Common handler")
	defer handlerSpan.Finish()
	var departureDate *string
	userTZ := handler.getTZByUserGeoID(queryParameters.GeoID, consts.MskTZ)
	queryParameters.UserTime = time.Now().In(userTZ)
	if isRelevantDate(queryParameters.DepartureDate, userTZ) {
		departureDate = ptr.String(queryParameters.DepartureDate.Format(timeformats.RFC3339Date))
	}
	return handler.responseBuilder.Build(queryParameters, landingParameters, fromSettlement, ctx, departureDate)
}

func (handler *CommonHandler) getTZByUserGeoID(userGeoID int, defaultTZ *time.Location) *time.Location {
	if settlement, found := handler.settlementRepository.GetByID(userGeoID); found {
		if settlement.TimeZone != "" {
			if loc, err := handler.locationRepository.LoadLocation(settlement.TimeZone); err == nil {
				return loc
			}
		}
		if region, found := handler.regionRepository.GetByID(settlement.RegionID); found {
			if loc, err := handler.locationRepository.LoadLocation(region.TimeZone); err == nil {
				return loc
			}
		}
	}
	return defaultTZ
}
