package handlers

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	flightsModel "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights"
	flightModels "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights/models"
)

type FlightHandler struct {
	appLogger                     log.Logger
	companyRepository             repositories.Company
	countryRepository             repositories.Country
	regionRepository              repositories.Region
	settlementRepository          repositories.Settlement
	settlementTimeZoneRepository  repositories.SettlementTimeZone
	stationToSettlementRepository repositories.StationToSettlement
	responseBuilder               building.FlightResponseBuilder

	landingBuilder         landings.FrontLandingBuilder
	bannedFlightRepository repositories.BannedFlight
	sharedFlightsClient    flights.Provider
}

func NewFlightHandler(
	appLogger log.Logger,
	companyRepository repositories.Company,
	countryRepository repositories.Country,
	regionRepository repositories.Region,
	settlementRepository repositories.Settlement,
	settlementTimeZoneRepository repositories.SettlementTimeZone,
	stationToSettlementRepository repositories.StationToSettlement,
	landingBuilder landings.FrontLandingBuilder,
	bannedFlightRepository repositories.BannedFlight,
	sharedFlightsClient flights.Provider,
	responseBuilder building.FlightResponseBuilder,
) *FlightHandler {
	return &FlightHandler{
		appLogger:                     appLogger,
		companyRepository:             companyRepository,
		countryRepository:             countryRepository,
		regionRepository:              regionRepository,
		settlementRepository:          settlementRepository,
		settlementTimeZoneRepository:  settlementTimeZoneRepository,
		stationToSettlementRepository: stationToSettlementRepository,
		responseBuilder:               responseBuilder,

		landingBuilder:         landingBuilder,
		bannedFlightRepository: bannedFlightRepository,
		sharedFlightsClient:    sharedFlightsClient,
	}
}

func (handler *FlightHandler) Handle(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
	pointFrom, pointTo models.Point,
	flightNumber string,
	landingParameters map[string]string,
) (*responses.FlightResponse, error) {
	now := time.Now()
	handlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "Flight handler")
	defer handlerSpan.Finish()
	defer func() { metrics.GlobalWizardMetrics().GetFlightRequestTimer().RecordDuration(time.Since(now)) }()
	wizardType := building.WizardTypeFlight
	wizardSubtype := building.WizardSubtypeExactFlight

	if helpers.IsDigit(flightNumber) {
		wizardSubtype = building.WizardSubtypeQuasiFlight
		landingParameters["utm_term"] = wizardSubtype.String()
	}

	props.SetSearchProp(ctx, "wizard_type", wizardType.String())
	props.SetSearchProp(ctx, "wizard_subtype", wizardSubtype.String())

	flightNumber, err := handler.normalizeFlightNumber(flightNumber)
	if err != nil {
		return nil, err
	}

	if handler.bannedFlightRepository.IsBanned(flightNumber) {
		return nil, building.ErrBannedFlight
	}

	userTZ := handler.settlementTimeZoneRepository.GetByGeoID(queryParameters.GeoID)
	if userTZ == nil {
		userTZ = consts.MskTZ
	}
	nowAware := time.Now().In(userTZ)

	departureDate := queryParameters.FlightDate
	if departureDate == nil {
		departureDate = queryParameters.DepartureDate
	}

	flightRequestDate := nowAware
	if departureDate != nil {
		flightRequestDate = time.Date(
			departureDate.Year(),
			departureDate.Month(),
			departureDate.Day(),
			0,
			0,
			0,
			0,
			userTZ,
		)
		if err = handler.validateFlightRequestDate(flightRequestDate, nowAware); err != nil {
			return nil, err
		}
	}

	utmMedium := "flight"
	utmCampaign := "without_date"
	if departureDate != nil {
		utmCampaign = "with_date"
	}

	props.SetContextProp(ctx, "utm_medium", utmMedium)
	props.SetContextProp(ctx, "utm_campaign", utmCampaign)

	landingParameters["utm_medium"] = utmMedium
	landingParameters["utm_campaign"] = utmCampaign

	var flightVariants []*responses.FlightVariant

	if helpers.IsDigit(flightNumber) {
		flightVariants, err = handler.buildFlightVariantsByDigitalNumber(ctx, flightNumber, queryParameters, departureDate, flightRequestDate, nowAware, landingParameters, pointFrom, pointTo)
		if err != nil {
			return nil, err
		}
		props.SetSearchProp(ctx, "searching_flights_by_digital_number", "1")
	} else {
		flightVariant, err := handler.buildFlightVariantByFullNumber(ctx, flightNumber, queryParameters, departureDate, flightRequestDate, nowAware, landingParameters, pointFrom, pointTo)
		if err != nil {
			return nil, err
		}
		flightVariants = append(flightVariants, flightVariant)
	}
	props.SetSearchProp(ctx, "flight_variants_count", len(flightVariants))

	variantForDeprecatedFormat := flightVariants[0]
	if queryParameters.Flags.DisableDeprecatedFlightResponseFormat() {
		variantForDeprecatedFormat = new(responses.FlightVariant)
	}

	return &responses.FlightResponse{
		Button:         variantForDeprecatedFormat.Button,
		Content:        variantForDeprecatedFormat.Content,
		Factors:        variantForDeprecatedFormat.Factors,
		GreenURL:       variantForDeprecatedFormat.GreenURL,
		Title:          variantForDeprecatedFormat.Title,
		Flags:          queryParameters.Flags,
		Params:         landingParameters,
		ReqID:          queryParameters.MainReqID,
		Type:           wizardType.String(),
		Subtype:        wizardSubtype.String(),
		FlightVariants: flightVariants,
	}, nil
}

func (handler *FlightHandler) buildFlightVariantsByDigitalNumber(
	ctx context.Context,
	flightNumber string,
	queryParameters *parameters.QueryParameters,
	departureDate *time.Time,
	flightRequestDate time.Time,
	nowAware time.Time,
	landingParameters map[string]string,
	pointFrom, pointTo models.Point,
) ([]*responses.FlightVariant, error) {
	digitalFlightNumber, err := strconv.Atoi(flightNumber)
	if err != nil {
		return nil, building.ErrBadFlightNumber
	}

	nv := queryParameters.NationalVersion
	if nv == "by" {
		nv = "ru"
	}

	flightsRequest := &flightModels.MultiRequest{
		FlightNumber:           digitalFlightNumber,
		NationalVersion:        nv,
		Date:                   flightRequestDate,
		ArrivedFlightsCount:    2,
		NotArrivedFlightsCount: 4,
		Context:                ctx,
		MainReqID:              queryParameters.MainReqID,
		ReqID:                  queryParameters.ReqID,
		JobID:                  queryParameters.JobID,
	}

	flightsResult, err := handler.sharedFlightsClient.GetFlightsMulti(flightsRequest)
	if err != nil {
		return nil, domain.NewWizardError(err.Error(), domain.SharedFlightsError)
	}

	var flightVariants []*responses.FlightVariant
	for _, companyFlights := range flightsResult {
		if len(companyFlights) == 0 {
			continue
		}

		flightVariant, err := handler.buildFlightVariant(
			ctx,
			queryParameters,
			landingParameters,
			companyFlights,
			departureDate,
			nowAware,
			pointFrom, pointTo,
		)
		if err != nil {
			continue
		}

		flightVariants = append(flightVariants, flightVariant)
	}

	if len(flightVariants) == 0 {
		return nil, building.ErrNoFlightVariants
	}
	return flightVariants, nil
}

func (handler *FlightHandler) buildFlightVariantByFullNumber(
	ctx context.Context,
	flightNumber string,
	queryParameters *parameters.QueryParameters,
	departureDate *time.Time,
	flightRequestDate time.Time,
	nowAware time.Time,
	landingParameters map[string]string,
	pointFrom, pointTo models.Point,
) (*responses.FlightVariant, error) {
	flightsRequest := &flightModels.Request{
		FlightNumber:           flightNumber,
		Date:                   flightRequestDate,
		ArrivedFlightsCount:    2,
		NotArrivedFlightsCount: 4,
		Context:                ctx,
		MainReqID:              queryParameters.MainReqID,
		ReqID:                  queryParameters.ReqID,
		JobID:                  queryParameters.JobID,
	}
	flightsResult, err := handler.sharedFlightsClient.GetFlights(flightsRequest)
	if err != nil {
		return nil, domain.NewWizardError(err.Error(), domain.SharedFlightsError)
	}

	if len(flightsResult) == 0 {
		return nil, building.ErrNoFlight
	}

	return handler.buildFlightVariant(
		ctx,
		queryParameters,
		landingParameters,
		flightsResult,
		departureDate,
		nowAware,
		pointFrom, pointTo,
	)
}

func (handler *FlightHandler) buildFlightVariant(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
	landingParameters map[string]string,
	flightsResult flightsModel.Flights,
	departureDate *time.Time,
	nowAware time.Time,
	pointFrom, pointTo models.Point,
) (*responses.FlightVariant, error) {
	flightNumber := flightsResult[0].Number

	if departureDate != nil {
		if len(filterFlightsByDepartureDate(flightsResult, *departureDate)) == 0 {
			return nil, building.ErrNoFlight
		}
	}

	tabs := building.BuildFlightTabs(flightsResult)

	var (
		defaultDate     *time.Time
		defaultTabIndex int
	)
	for index, tab := range tabs {
		var applyDefaultDate bool
		if departureDate != nil {
			applyDefaultDate = tab.Date.After(*departureDate)
		} else {
			flight := tab.Flights[len(tab.Flights)-1]
			applyDefaultDate = flight.PointTo.GetEstimatedTime().Sub(nowAware) >= -2*time.Hour
		}
		if applyDefaultDate {
			defaultDate = &tab.Date
			defaultTabIndex = index
			break
		}
	}

	if defaultDate == nil {
		defaultTabIndex = len(tabs) - 1
		defaultDate = &tabs[defaultTabIndex].Date
	}

	defaultFlight := tabs[defaultTabIndex].Flights[0]

	var defaultFlightPointFrom models.Point
	var defaultFlightPointTo models.Point
	if cityFrom, ok := handler.settlementRepository.GetByID(defaultFlight.PointFrom.Airport.SettlementID); ok {
		defaultFlightPointFrom = cityFrom
	} else {
		defaultFlightPointFrom = defaultFlight.PointFrom.Airport
	}
	if cityTo, ok := handler.settlementRepository.GetByID(defaultFlight.PointTo.Airport.SettlementID); ok {
		defaultFlightPointTo = cityTo
	} else {
		defaultFlightPointTo = defaultFlight.PointTo.Airport
	}

	if !handler.checkCorrectDepartureAndArrivalPoints(
		pointFrom,
		pointTo,
		defaultFlight.PointFrom.Airport,
		defaultFlight.PointTo.Airport,
	) {
		return nil, building.ErrBadFlightPoints
	}

	tabs = building.GetDisplayedTabs(defaultTabIndex, tabs)

	props.SetSearchProp(ctx, "flightStatus", defaultFlight.StatusInfo.Status)
	props.SetSearchProp(ctx, "withCheckInDesks", withCheckInDesks(defaultFlight.StatusInfo.CheckInDesks))

	return handler.responseBuilder.Build(
		tabs,
		defaultFlightPointFrom, defaultFlightPointTo,
		flightNumber,
		queryParameters,
		nowAware,
		*defaultDate,
		defaultFlight,
		landingParameters,
		getCompanyCode(&flightsResult[0].Company),
	)
}

func (handler *FlightHandler) normalizeFlightNumber(flightNumber string) (string, error) {
	if helpers.IsDigit(flightNumber) {
		return flightNumber, nil
	}
	splittedFlightNumber := strings.Split(flightNumber, " ")
	if len(splittedFlightNumber) != 2 {
		return "", building.ErrBadFlightNumber
	}

	companyCode, companyFlightNumber := splittedFlightNumber[0], splittedFlightNumber[1]
	companyCode = strings.ToUpper(companyCode)

	company, found := handler.companyRepository.GetByIata(companyCode)
	if !found {
		company, _ = handler.companyRepository.GetBySirena(companyCode)
	}

	if company == nil {
		return flightNumber, nil
	}

	return fmt.Sprintf("%s %s", getCompanyCode(company), companyFlightNumber), nil
}

func (handler *FlightHandler) checkCorrectDepartureAndArrivalPoints(
	pointFrom, pointTo models.Point,
	airportDeparture, airportArrival *models.Station,
) bool {
	checkedPoints := make(models.SetOfPoints)
	if pointFrom != nil {
		checkedPoints.Add(pointFrom)
	}
	if pointTo != nil {
		checkedPoints.Add(pointTo)
	}

	allowedPoints := make(models.SetOfPoints)
	allowedPoints.Add(nil)

	for _, point := range []*models.Station{airportDeparture, airportArrival} {
		allowedPoints.Add(point)

		settlement, _ := handler.settlementRepository.GetByID(point.SettlementID)
		if settlement != nil {
			allowedPoints.Add(settlement)
		}

		region, _ := handler.regionRepository.GetByID(airportDeparture.RegionID)
		if region != nil {
			allowedPoints.Add(region)
		}

		country, _ := handler.countryRepository.GetByID(airportDeparture.CountryID)
		if country != nil {
			allowedPoints.Add(country)
		}

		settlements := handler.stationToSettlementRepository.GetSettlementsByStationID(airportDeparture.SettlementID)
		for _, settlement := range settlements {
			allowedPoints.Add(settlement)
		}
	}
	return checkedPoints.IsSubset(allowedPoints)
}

func (handler *FlightHandler) validateFlightRequestDate(flightRequestDate time.Time, nowAware time.Time) error {
	year := 365 * 24 * time.Hour
	if flightRequestDate.Before(nowAware.Add(-year)) || flightRequestDate.After(nowAware.Add(year)) {
		return building.ErrInvalidFlightDate
	}
	return nil
}

func getCompanyCode(company *models.Company) string {
	code := company.Iata
	if code == "" {
		code = company.SirenaID
	}
	return code
}

func filterFlightsByDepartureDate(srcFlights flightsModel.Flights, date time.Time) flightsModel.Flights {
	var filtered flightsModel.Flights
	for _, flight := range srcFlights {
		if flight.PointFrom.ScheduledTime.Truncate(24 * time.Hour).Equal(date) {
			filtered = append(filtered, flight)
		}
	}
	return filtered
}

func withCheckInDesks(checkInDesks string) int {
	if len(checkInDesks) > 0 {
		return 1
	}
	return 0
}
