package handlers

import (
	"context"
	"strings"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/mappers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/search"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	appfeatureflag "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/featureflag"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/personalization"
)

type PointToPointHandler struct {
	logger                       log.Logger
	settlementRepository         repositories.Settlement
	pointParser                  point.IParser
	regionRepository             repositories.Region
	minPriceToSearchResultMapper *mappers.WizardToSearchResultMapper
	responseBuilder              *building.PointToPointResponseBuilder
	locationRepository           repositories.CachedLocation
	resultProvider               *search.ResultProvider
	enablePersonalization        bool
	featureFlagStorage           *appfeatureflag.Storage
}

func NewPointToPointHandler(
	logger log.Logger,
	settlementRepository repositories.Settlement,
	regionRepository repositories.Region,
	pointParser point.IParser,
	resultProvider *search.ResultProvider,
	locationRepository repositories.CachedLocation,
	minPriceToSearchResultMapper *mappers.WizardToSearchResultMapper,
	responseBuilder *building.PointToPointResponseBuilder,
	enablePersonalization bool,
	featureFlagStorage *appfeatureflag.Storage,
) *PointToPointHandler {
	return &PointToPointHandler{
		logger:                       logger,
		settlementRepository:         settlementRepository,
		pointParser:                  pointParser,
		regionRepository:             regionRepository,
		resultProvider:               resultProvider,
		locationRepository:           locationRepository,
		minPriceToSearchResultMapper: minPriceToSearchResultMapper,
		responseBuilder:              responseBuilder,
		enablePersonalization:        enablePersonalization,
		featureFlagStorage:           featureFlagStorage,
	}
}

func (handler *PointToPointHandler) Handle(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
	fromPoint, toPoint models.Point,
	handlerType,
	utmCampaign string,
	landingParameters map[string]string,
	wizardType building.WizardType,
) (*responses.PointToPointResponse, error) {
	now := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().GetPointToPointRequestTimer().RecordDuration(time.Since(now))
		if queryParameters.IsDynamic() {
			metrics.GlobalWizardMetrics().GetDynamicRequestTimer().RecordDuration(time.Since(now))
		} else {
			metrics.GlobalWizardMetrics().GetStaticRequestTimer().RecordDuration(time.Since(now))
		}
	}()
	handlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "Point-to-point handler")
	defer handlerSpan.Finish()
	ctx = handler.sleepIfNecessary(ctx, queryParameters)

	departureDate := queryParameters.DepartureDate
	returnDate := queryParameters.ReturnDate
	toCrimea := isCrimeaText(queryParameters.ToText)

	props.SetSearchProp(
		ctx,
		"there_is_an_airport_among_points",
		!helpers.IsNil(fromPoint) && fromPoint.GetPointType() == models.PointTypeStation ||
			!helpers.IsNil(toPoint) && toPoint.GetPointType() == models.PointTypeStation,
	)

	fromSettlement, err := handler.pointParser.PointToSettlement(fromPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic())
	if err != nil {
		return nil, err
	}
	toSettlement, err := handler.pointParser.PointToSettlement(toPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic())
	if err != nil {
		return nil, err
	}
	props.SetSearchProp(ctx, "from_settlement_id", fromSettlement.ID)
	props.SetSearchProp(ctx, "to_settlement_id", toSettlement.ID)

	if err := handler.validateSettlements(fromSettlement, toSettlement); err != nil {
		return nil, err
	}

	userTZ := handler.getTZByUserGeoID(queryParameters.GeoID, consts.MskTZ)
	queryParameters.UserTime = time.Now().In(userTZ)

	// at this point either both dates are nil or only return is nil or both have some values
	if queryParameters.IsDynamic() {
		// dynamic requests require strict validations
		if departureDate == nil {
			return nil, domain.NewWizardError("departure date is required for dynamic requests", domain.NoDate)
		}
		if !isRelevantDate(departureDate, userTZ) {
			return nil, domain.NewWizardError("departure date is in the past or too far in the future", domain.InvalidDate)
		}
		if returnDate != nil && !isRelevantDate(returnDate, userTZ) {
			return nil, domain.NewWizardError("return date is in the past or too far in the future", domain.InvalidDate)
		}
		if isReturnBeforeDeparture(departureDate, queryParameters.ReturnDate) {
			return nil, domain.NewWizardError("return date is before departure date", domain.InvalidDate)
		}
	} else {
		// static requests permit invalid dates, we just nil them
		if !isRelevantDate(departureDate, userTZ) {
			departureDate = nil
		}
		if departureDate == nil || !isRelevantDate(returnDate, userTZ) || isReturnBeforeDeparture(departureDate, returnDate) {
			queryParameters.ReturnDate = nil
		}
	}

	if !queryParameters.IsDynamic() && departureDate != nil {
		props.SetSearchProp(ctx, "date_from_gar", departureDate.Format("2006-01-02"))
	}

	if !queryParameters.IsDynamic() && helpers.IsUkraine(fromSettlement, toSettlement) {
		return nil, domain.NewWizardError("unavailable direction", domain.BadArguments)
	}
	if !queryParameters.IsDynamic() && handler.enablePersonalization {
		personalSearchResponse, err := handler.resultProvider.GetPersonalSearchResult(ctx, queryParameters)
		if err != nil {
			handler.logger.Error("Personalization error", log.Error(err))
		}

		personalSearchMapper := personalization.NewPersonalSearchMapper()
		personalSearchSuggest := personalSearchMapper.Map(fromSettlement, toSettlement, personalSearchResponse)

		var dateFromSearchProp = ""
		if !personalSearchSuggest.DepartureDate.IsZero() {
			dateFromSearchProp = personalSearchSuggest.DepartureDate.Format("2006-01-02")
			metrics.GlobalWizardMetrics().GetPersonalizationActivationCounter().Inc()
		}
		props.SetSearchProp(ctx, "date_from_personalization", dateFromSearchProp)

		if departureDate == nil &&
			isRelevantDate(&personalSearchSuggest.DepartureDate, userTZ) &&
			(queryParameters.Flags.PersonalizedDefaultDay() || queryParameters.Flags.PersonalizedDefaultDayOrMinPriceDate()) {

			departureDate = &personalSearchSuggest.DepartureDate
		}
	}

	wizardResult, err := handler.resultProvider.GetWizardResult(queryParameters, departureDate, fromSettlement, toSettlement, now, ctx)
	if err != nil {
		return nil, err
	}

	var personalizationCacheMiss = wizardResult == nil || wizardResult.DateForward != search.ConvertDateToDays(departureDate)
	props.SetSearchProp(ctx, "personalization_cache_miss", personalizationCacheMiss)
	if !personalizationCacheMiss {
		metrics.GlobalWizardMetrics().GetPersonalizationCacheHitsCounter().Inc()
	}

	searchResult := handler.minPriceToSearchResultMapper.Map(wizardResult, queryParameters, ctx)

	response, err := handler.responseBuilder.Build(
		ctx,
		searchResult,
		handlerType,
		utmCampaign,
		wizardType,
		queryParameters,
		landingParameters,
		fromSettlement, toSettlement,
		toCrimea,
		handler.featureFlagStorage.BoyIsEnabled(
			featureflag.NewABFlagsKV(
				queryParameters.Flags,
			),
		),
	)
	return response, err
}

func (handler *PointToPointHandler) sleepIfNecessary(ctx context.Context, queryParameters *parameters.QueryParameters) context.Context {
	operationName := "Sleep static"
	sleepDuration := queryParameters.Flags.SleepPPStaticDuration()
	if queryParameters.IsDynamic() {
		operationName = "Sleep dynamic"
		sleepDuration = queryParameters.Flags.SleepPPDynamicDuration()
	}
	if sleepDuration > 0 {
		var sleepSpan opentracing.Span
		sleepSpan, ctx = opentracing.StartSpanFromContext(ctx, operationName)
		defer sleepSpan.Finish()
		time.Sleep(sleepDuration)
	}
	return ctx
}

func isCrimeaText(text string) bool {
	return strings.ToLower(text) == "крым"
}

func (handler *PointToPointHandler) validateSettlements(fromSettlement, toSettlement *models.Settlement) error {
	if fromSettlement == toSettlement {
		return domain.NewWizardError("from_settlement and to_settlement are identical", domain.SamePoint)
	}

	if !handler.settlementRepository.IsAviaSettlement(fromSettlement.ID) {
		return domain.NewWizardError("from_settlement has no airports", domain.NoAirport)
	}
	if !handler.settlementRepository.IsAviaSettlement(toSettlement.ID) {
		return domain.NewWizardError("to_settlement has no airports", domain.NoAirport)
	}

	if handler.pointParser.PointsIntersect(fromSettlement, toSettlement) {
		return domain.NewWizardError("from_settlement and to_settlement have the same airport(s).", domain.SameAirports)
	}

	return nil
}

func (handler *PointToPointHandler) getTZByUserGeoID(userGeoID int, defaultTZ *time.Location) *time.Location {
	if settlement, found := handler.settlementRepository.GetByID(userGeoID); found {
		if settlement.TimeZone != "" {
			if loc, err := handler.locationRepository.LoadLocation(settlement.TimeZone); err == nil {
				return loc
			}
		}
		if region, found := handler.regionRepository.GetByID(settlement.RegionID); found {
			if loc, err := handler.locationRepository.LoadLocation(region.TimeZone); err == nil {
				return loc
			}
		}
	}
	return defaultTZ
}

// Checks whether date makes sense for users' timezone (it is not in the past and not too far in the future)
func isRelevantDate(date *time.Time, userTZ *time.Location) bool {
	if date == nil {
		return false
	}

	userDate := helpers.TruncateToDate(time.Now().In(userTZ))
	dateInUserTZ := helpers.ReplaceLocation(*date, userTZ)

	return !userDate.After(dateInUserTZ) && !dateInUserTZ.After(userDate.Add(360*24*time.Hour))
}

func isReturnBeforeDeparture(departureDate, returnDate *time.Time) bool {
	if departureDate == nil || returnDate == nil {
		return false
	}
	return returnDate.Before(*departureDate)
}

func (handler *PointToPointHandler) choosePointsAndDates(
	fromPoint, toPoint models.Point,
	fromSettlement, toSettlement *models.Settlement,
	queryParameters *parameters.QueryParameters,
) (models.Point, models.Point) {
	if handler.shouldCastPointToSettlement(fromPoint, fromSettlement, queryParameters) {
		fromPoint = fromSettlement
	}
	if handler.shouldCastPointToSettlement(toPoint, toSettlement, queryParameters) {
		toPoint = toSettlement
	}
	// TODO: RASPTICKETS-17580 разобраться с логикой доставания из базы в случае разных запросов
	// if departureDate != nil {
	//	return fromPoint, toPoint
	// }
	return fromPoint, toPoint
}

func (handler *PointToPointHandler) shouldCastPointToSettlement(point models.Point, settlementFromPoint *models.Settlement, queryParameters *parameters.QueryParameters) bool {
	return point.GetPointType() != models.PointTypeStation ||
		len(handler.settlementRepository.GetStationIDs(settlementFromPoint.ID)) == 1 ||
		!queryParameters.Flags.EnableSearchToAirports() ||
		!queryParameters.IsTouchDevice()
}
