package handlers

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	pointParse "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

type RouteHandler struct {
	appLogger              log.Logger
	pointParseLogger       *pointParse.Logger
	flight                 *FlightHandler
	pointParser            point.IParser
	pointToPointHander     *PointToPointHandler
	commonHandler          *CommonHandler
	flightNumberNormalizer *flights.FlightNumberNormalizer
}

func NewRouteHandler(
	appLogger log.Logger,
	pointParseLogger *pointParse.Logger,
	flight *FlightHandler,
	flightPointParser point.IParser,
	pointToPointHandler *PointToPointHandler,
	commonHandler *CommonHandler,
	flightNumberNormalizer *flights.FlightNumberNormalizer,
) *RouteHandler {
	return &RouteHandler{
		appLogger:              appLogger,
		pointParseLogger:       pointParseLogger,
		flight:                 flight,
		pointParser:            flightPointParser,
		pointToPointHander:     pointToPointHandler,
		commonHandler:          commonHandler,
		flightNumberNormalizer: flightNumberNormalizer,
	}
}

func (handler *RouteHandler) Handle(ctx context.Context, queryParameters *parameters.QueryParameters) (responses.HandlerResponse, error) {
	routeHandlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "Route handler")
	defer routeHandlerSpan.Finish()
	var err error

	if err = handler.checkZoneAvailability(queryParameters); err != nil {
		return nil, err
	}

	normalizedFlightNumber := handler.flightNumberNormalizer.Normalize(queryParameters.FlightNumber, queryParameters.Company)

	props.SetContextProp(ctx, "serp_uuid", queryParameters.SerpUUID)

	utmSource := fmt.Sprintf("wizard_%s", queryParameters.NationalVersion)
	if queryParameters.Flags.UseUnisearch() {
		utmSource = fmt.Sprintf("unisearch_%s", queryParameters.NationalVersion)
	}

	props.SetContextProp(ctx, "utm_source", utmSource)

	var landingParams map[string]string
	if queryParameters.Flags.UseUnisearch() {
		landingParams = map[string]string{
			"lang":           queryParameters.Lang.String(),
			"utm_source":     utmSource,
			"unisearchReqId": queryParameters.MainReqID,
			"req_id":         queryParameters.ReqID,
			"serp_uuid":      queryParameters.SerpUUID,
		}
	} else {
		landingParams = map[string]string{
			"lang":        queryParameters.Lang.String(),
			"utm_source":  utmSource,
			"wizardReqId": queryParameters.MainReqID,
			"req_id":      queryParameters.ReqID,
			"serp_uuid":   queryParameters.SerpUUID,
		}
	}

	flightFromPoint, flightToPoint, err := handler.pointParser.ParseFlightPoints(queryParameters, queryParameters.NationalVersion, ctx)
	if err != nil {
		return nil, err
	}
	ppFromPoint, ppToPoint, ppFromSettlement, ppToSettlement, err := handler.pointParser.ParsePointToPoint(ctx, queryParameters)
	if err != nil {
		return nil, err
	}

	var handlerResponse responses.HandlerResponse
	if shouldHandleAsFlightRequest(ctx, handler.appLogger, normalizedFlightNumber, queryParameters) {
		handler.logParsedPoints(queryParameters, flightFromPoint, flightToPoint)
		handlerResponse, err = handler.flight.Handle(
			ctx, queryParameters,
			flightFromPoint, flightToPoint,
			normalizedFlightNumber,
			landingParams,
		)
		if err != nil {
			return nil, err
		}
	} else if ppToSettlement != nil {
		// TODO: sleep flag
		handler.logParsedPoints(queryParameters, ppFromPoint, ppToPoint)
		utmMedium := chooseUtmMedium(queryParameters)
		wizardType := building.WizardTypePP
		if queryParameters.Company != nil {
			wizardType = building.WizardTypePPAirline
		}
		props.SetSearchProp(ctx, "wizard_type", wizardType.String())
		utmCampaign := chooseUtmCampaign(ppToPoint, queryParameters)
		handlerType := utmCampaign
		props.SetSearchProp(ctx, "handler_type", handlerType)

		if !queryParameters.IsDynamic() && queryParameters.Company != nil {
			availableServices := containers.NewSetOfString("desktop", "tablet")
			if !availableServices.Contains(queryParameters.Device) && !queryParameters.Flags.EnablePPCompanies() {
				return nil, domain.NewWizardError(
					fmt.Sprintf(
						"available devices: %v. use enable_pp_companies=1 flag to enable on other devices",
						availableServices.ToSlice(),
					),
					domain.P2PCompany,
				)
			}
			utmMedium = "pp_airline"
		}
		if !queryParameters.Flags.UseUnisearch() {
			landingParams["from"] = "aviawizard_pp"
		}
		landingParams["utm_medium"] = utmMedium
		landingParams["utm_campaign"] = utmCampaign
		passengersJSON, _ := json.Marshal(queryParameters.Passengers())
		landingParams["passengers"] = strings.ReplaceAll(strings.ToLower(string(passengersJSON)), "\"", "'")

		props.SetContextProp(ctx, "utm_campaign", utmCampaign)
		props.SetContextProp(ctx, "utm_medium", utmMedium)

		handlerResponse, err = handler.pointToPointHander.Handle(
			ctx,
			queryParameters,
			ppFromPoint,
			ppToPoint,
			handlerType, utmCampaign,
			landingParams,
			wizardType,
		)

		if err != nil {
			return nil, err
		}
		props.SetSearchProp(ctx, "direct_flight", queryParameters.DirectFlight)
	} else {
		handler.logParsedPoints(queryParameters, ppFromPoint, ppToPoint)
		utmMedium := "common"
		utmCampaign := "common"
		landingParams["form"] = "waviablank"
		if queryParameters.Company != nil {
			utmMedium = "common_airline"
			utmCampaign = "common_airline"
			landingParams["form"] = "waviablank_airline"
		}
		landingParams["utm_campaign"] = utmCampaign
		landingParams["utm_medium"] = utmMedium

		props.SetContextProp(ctx, "utm_medium", utmMedium)
		props.SetContextProp(ctx, "utm_campaign", utmCampaign)

		handlerResponse, err = handler.commonHandler.Handle(queryParameters, ppFromSettlement, landingParams, ctx)

		if err != nil {
			return nil, err
		}
	}

	addFiltersToContext(ctx, queryParameters, handlerResponse)

	return handlerResponse, nil
}

func addFiltersToContext(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
	response responses.HandlerResponse,
) {
	appliedFilters := response.GetAppliedFilters()
	if queryParameters.Company != nil &&
		!(queryParameters.Flags.FallbackAirlineFilter() && !(appliedFilters.Airlines == nil || len(*appliedFilters.Airlines) == 0)) {
		props.SetContextProp(ctx, "company", queryParameters.CompanyID)
	}
	if queryParameters.DirectFlight && !(queryParameters.Flags.FallbackDirectFilter() && appliedFilters.Transfer == nil) {
		props.SetContextProp(ctx, "direct_flight", queryParameters.DirectFlight)
	}
}

func chooseUtmCampaign(ppToPoint models.Point, queryParameters *parameters.QueryParameters) string {
	// TODO: RASPTICKETS-17580
	// if queryParameters.AviaDynamic.Filters.Airport != nil &&
	// 	(queryParameters.AviaDynamic.Filters.Airport.ForwardDeparutre != nil ||
	// 		queryParameters.AviaDynamic.Filters.Airport.ForwardArrival != nil) {
	// 	return "airport"
	// }
	switch ppToPoint.(type) {
	case *models.Country:
		return "country"
	case *models.Region:
		return "region"
	case *models.Settlement:
		return "city"
	case *models.Station:
		return "airport"
	}
	return "unknown"
}

func chooseUtmMedium(queryParameters *parameters.QueryParameters) string {
	utmMedium := "pp"
	if utmMediumValue, ok := queryParameters.Context().UtmMedium(); ok && utmMediumValue != nil {
		utmMedium = *utmMediumValue
	}
	return utmMedium
}

func (handler *RouteHandler) checkZoneAvailability(queryParameters *parameters.QueryParameters) error {
	if !settings.IsAllowedLanguage(queryParameters.Lang) {
		return domain.NewWizardError("Bad lang", domain.BadLang)
	}
	if queryParameters.Lang == consts.LangKK && !queryParameters.Flags.KKEnabled() {
		return domain.NewWizardError("kk is supported under experiment only", domain.BadLang)
	}

	if !settings.IsAllowedTld(queryParameters.Tld) {
		return domain.NewWizardError(fmt.Sprintf("bad tld: %s", queryParameters.Tld), domain.BadTld)
	}

	if !settings.IsAllowedNationalVersion(queryParameters.NationalVersion) {
		return domain.NewWizardError(
			fmt.Sprintf("National version %s is not supported", queryParameters.NationalVersion),
			domain.BadNationalVersion,
		)
	}
	return nil
}

func (handler *RouteHandler) logParsedPoints(queryParameters *parameters.QueryParameters, fromPoint, toPoint models.Point) {
	var fromPointKey, toPointKey *string
	if !helpers.IsNil(fromPoint) {
		fromPointKey = ptr.String(fromPoint.GetPointKey())
	}
	if !helpers.IsNil(toPoint) {
		toPointKey = ptr.String(toPoint.GetPointKey())
	}
	handler.pointParseLogger.Log(pointParse.NewRecord(queryParameters, fromPointKey, toPointKey))
}

func shouldHandleAsFlightRequest(
	ctx context.Context,
	appLogger log.Logger,
	flightNumber string,
	queryParameters *parameters.QueryParameters,
) bool {
	if flightNumber == "" {
		appLogger.Info("flightNumber is empty", log.String("job_id", queryParameters.JobID))
		return false
	}

	if helpers.IsDigit(flightNumber) && queryParameters.QueryHasDate {
		appLogger.Info("flightNumber might be a date", log.String("job_id", queryParameters.JobID))
		return false
	}

	props.SetSearchProp(ctx, "flight_number_is_digital", helpers.IsDigit(flightNumber))
	return true
}
