package handlers

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/mappers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/search"
)

type VariantsInfoHandler struct {
	logger                       log.Logger
	pointParser                  point.IParser
	minPriceToSearchResultMapper *mappers.WizardToSearchResultMapper
	responseBuilder              *building.VariantsInfoResponseBuilder
	resultProvider               *search.ResultProvider
}

func NewVariantsInfoHandler(
	logger log.Logger,
	pointParser point.IParser,
	resultProvider *search.ResultProvider,
	minPriceToSearchResultMapper *mappers.WizardToSearchResultMapper,
	responseBuilder *building.VariantsInfoResponseBuilder,
) *VariantsInfoHandler {
	return &VariantsInfoHandler{
		logger:                       logger,
		pointParser:                  pointParser,
		resultProvider:               resultProvider,
		minPriceToSearchResultMapper: minPriceToSearchResultMapper,
		responseBuilder:              responseBuilder,
	}
}

func (handler *VariantsInfoHandler) Handle(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
	landingParameters map[string]string,
) (*responses.VariantsInfoResponse, error) {
	now := time.Now()
	handlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "variants info handler")
	defer handlerSpan.Finish()

	departureDate := queryParameters.DepartureDate
	toCrimea := isCrimeaText(queryParameters.ToText)

	pointFrom, pointTo, _, _, err := handler.pointParser.ParsePointToPoint(ctx, queryParameters)
	if err != nil {
		return nil, err
	}

	fromSettlement, err := handler.pointParser.PointToSettlement(pointFrom, queryParameters.NationalVersion, !queryParameters.IsDynamic())
	if err != nil {
		return nil, err
	}
	toSettlement, err := handler.pointParser.PointToSettlement(pointTo, queryParameters.NationalVersion, !queryParameters.IsDynamic())
	if err != nil {
		return nil, err
	}
	if fromSettlement == toSettlement {
		return nil, domain.NewWizardError("fromSettlement and toSettlement are identical", domain.SamePoint)
	}
	if handler.pointParser.PointsIntersect(fromSettlement, toSettlement) {
		return nil, domain.NewWizardError("fromSettlement and toSettlement have the same airport(s).", domain.SameAirports)
	}

	// at this point both dates are nil, therefore no validations are applied

	wizardResult, err := handler.resultProvider.GetWizardResult(
		queryParameters,
		departureDate,
		fromSettlement,
		toSettlement,
		now,
		ctx,
	)
	if err != nil {
		return nil, err
	}

	searchResult := handler.minPriceToSearchResultMapper.Map(wizardResult, queryParameters, ctx)
	response, err := handler.responseBuilder.Build(
		ctx,
		searchResult,
		queryParameters,
		landingParameters,
		fromSettlement,
		toSettlement,
		toCrimea,
	)
	return response, err
}
