package handlers

import (
	"context"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	domainResponses "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	queryLog "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/query"
	responseLog "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/response"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

type WizardHandler struct {
	appLogger              log.Logger
	queryLogger            *queryLog.Logger
	responseLogger         *responseLog.Logger
	queryParametersParser  parameters.IQueryParametersParser
	routeHandler           *RouteHandler
	settings               *settings.Settings
	flightNumberNormalizer *flights.FlightNumberNormalizer
}

func (handler *WizardHandler) Handle(request *http.Request) (
	handlerResponse domainResponses.HandlerResponse,
	errorResponse *domainResponses.ErrorResponse,
) {
	startTime := time.Now()
	ctx := request.Context()
	queryParameters, err := handler.queryParametersParser.Parse(request)
	if err != nil {
		return nil, domainResponses.NewErrorResponseFromError(err, nil, nil)
	}
	var cancelFunc func()
	if queryParameters.IsDynamic() {
		ctx, cancelFunc = context.WithTimeout(ctx, handler.settings.MaxDynamicRequestDuration-time.Since(startTime))
	} else {
		ctx, cancelFunc = context.WithTimeout(ctx, handler.settings.MaxStaticRequestDuration-time.Since(startTime))
	}
	defer cancelFunc()

	handler.queryLogger.Log(
		queryLog.NewRecord(
			queryParameters,
			consts.PageTypeSearch,
			handler.flightNumberNormalizer.Normalize(queryParameters.FlightNumber, queryParameters.Company),
		),
	)

	defer func() {
		responseStatus := "OK"
		var traceback *string
		if r := recover(); r != nil {
			traceback = ptr.String(helpers.GetTraceback())
			err = domain.NewWizardError(*traceback, domain.Fail)
			handler.appLogger.Error("An error occured during handling request", log.String("job_id", queryParameters.JobID))
		}
		if err != nil {
			errorMessage := ""
			switch e := err.(type) {
			case *domain.WizardError:
				responseStatus = e.Code
				errorMessage = e.Message
			default:
				responseStatus = "unknown"
				errorMessage = e.Error()
			}
			props.SetSearchProp(ctx, "error_text", errorMessage)
			props.SetSearchProp(ctx, "error", responseStatus)

			metrics.GlobalWizardMetrics().GetErrorCounter(responseStatus).Inc()
			errorResponse = domainResponses.NewErrorResponseFromError(
				err,
				props.SearchPropsGetAll(ctx),
				props.ContextPropsGetAll(ctx),
			)
		} else {
			handlerResponse.SetSearchProps(props.SearchPropsGetAll(ctx))
			handlerResponse.SetContext(props.ContextPropsGetAll(ctx))
		}
		var wizardType *string
		if !helpers.IsNil(handlerResponse) {
			wizardType = handlerResponse.GetType()
		} else if ht, ok := props.SearchPropsGet(ctx, "handler_type"); ok {
			wizardType = ptr.String(ht.(string))
		} else if wt, ok := props.SearchPropsGet(ctx, "wizard_type"); ok {
			wizardType = ptr.String(wt.(string))
		}
		handler.responseLogger.Log(
			responseLog.NewRecord(
				queryParameters,
				responseStatus,
				wizardType,
				getWizardSubtype(handlerResponse),
				getFlightStatus(ctx),
				getDefaultFlight(handlerResponse),
				traceback,
				getWithFilters(handlerResponse),
				props.SearchPropsGetAll(ctx),
				time.Since(startTime).Seconds(),
			),
		)
	}()
	handlerResponse, err = handler.routeHandler.Handle(ctx, queryParameters)
	return
}

func getWithFilters(response domainResponses.HandlerResponse) bool {
	if helpers.IsNil(response) {
		return false
	}
	r, ok := response.(*domainResponses.PointToPointResponse)
	return ok && r.FiltersDefaults != nil
}

func getFlightStatus(ctx context.Context) *string {
	if flightStatus, ok := props.SearchPropsGet(ctx, "flightStatus"); ok {
		if res, ok := flightStatus.(string); ok {
			return ptr.String(res)
		}
	}
	return nil
}

func getWizardSubtype(response domainResponses.HandlerResponse) *string {
	if helpers.IsNil(response) {
		return nil
	}
	if resp, ok := response.(*domainResponses.FlightResponse); ok {
		return ptr.String(resp.Subtype)
	}
	return nil
}

func getDefaultFlight(response domainResponses.HandlerResponse) *domainResponses.Flight {
	if helpers.IsNil(response) {
		return nil
	}
	if resp, ok := response.(*domainResponses.FlightResponse); ok {
		for i := 0; i < len(resp.Content.Tabs); i++ {
			if resp.Content.Tabs[i].Default {
				return &resp.Content.Tabs[i].Flights[0]
			}
		}
	}
	return nil
}

func NewWizardHandler(
	appLogger log.Logger,
	queryLogger *queryLog.Logger,
	responseLogger *responseLog.Logger,
	queryParametersParser parameters.IQueryParametersParser,
	routeHandler *RouteHandler,
	settings *settings.Settings,
	flightNumberNormalizer *flights.FlightNumberNormalizer,
) *WizardHandler {
	return &WizardHandler{
		appLogger:              appLogger,
		queryLogger:            queryLogger,
		responseLogger:         responseLogger,
		queryParametersParser:  queryParametersParser,
		routeHandler:           routeHandler,
		settings:               settings,
		flightNumberNormalizer: flightNumberNormalizer,
	}
}
