package landings

import (
	"crypto/hmac"
	"crypto/md5"
	"encoding/hex"
	"fmt"
	"net/url"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/translations"
)

type FrontLandingBuilder struct {
	queryParameters           *parameters.QueryParameters
	settings                  *settings.Settings
	params                    stringMap
	translatedTitleRepository repositories.TranslatedTitle
}

func NewFrontLandingBuilder(settings *settings.Settings, translatedTitleRepository repositories.TranslatedTitle) FrontLandingBuilder {
	return FrontLandingBuilder{
		settings:                  settings,
		params:                    make(stringMap),
		translatedTitleRepository: translatedTitleRepository,
	}
}

func (builder FrontLandingBuilder) WithQueryParameters(queryParameters *parameters.QueryParameters) FrontLandingBuilder {
	return FrontLandingBuilder{
		queryParameters:           queryParameters,
		settings:                  builder.settings,
		translatedTitleRepository: builder.translatedTitleRepository,
		params:                    copyMap(builder.params),
	}
}

func (builder FrontLandingBuilder) WithParams(params stringMap) FrontLandingBuilder {
	return FrontLandingBuilder{
		queryParameters:           builder.queryParameters,
		settings:                  builder.settings,
		params:                    mergeMap(builder.params, params),
		translatedTitleRepository: builder.translatedTitleRepository,
	}
}

func (builder FrontLandingBuilder) FlightMorda() *Landing {
	return NewLanding(fmt.Sprintf("%s/", builder.base(consts.DontUseTravelPortal)))
}

func (builder FrontLandingBuilder) Morda(useTravelPortal consts.TravelPortalUsage) *Landing {
	return NewLanding(fmt.Sprintf("%s/", builder.base(useTravelPortal)))
}

func (builder FrontLandingBuilder) SearchResult(
	fromSettlement, toSettlement *models.Settlement,
	fromName, toName string,
	departureDate, returnDate *time.Time,
	passengers *dynamic.Passengers,
) *Landing {
	params := map[string]string{
		"fromId":         fromSettlement.GetPointKey(),
		"toId":           toSettlement.GetPointKey(),
		"fromName":       fromName,
		"toName":         toName,
		"when":           departureDate.Format("2006-01-02"),
		"adult_seats":    strconv.Itoa(passengers.Adults),
		"children_seats": strconv.Itoa(passengers.Children),
		"infant_seats":   strconv.Itoa(passengers.Infants),
		"return_date":    "",
	}
	if returnDate != nil {
		params["return_date"] = returnDate.Format("2006-01-02")
	}
	urlTemplate := "%s/search/"
	useTravelPortal := builder.queryParameters.UseTravelPortal()
	if useTravelPortal {
		params["klass"] = "economy"
		urlTemplate = "%s/avia/search/result/"
	}
	return NewLanding(fmt.Sprintf(urlTemplate, builder.base(useTravelPortal))).WithParams(params)
}

func (builder *FrontLandingBuilder) Redirect(variant *responses.Variant) *Landing {
	tariffKey := fmt.Sprintf("%d%s", int32(variant.Tariff.Price), variant.Tariff.Currency)
	hasher := hmac.New(md5.New, []byte(builder.settings.RedirectSecretKey))
	hasher.Write([]byte(tariffKey))
	hashedTariffKey := hex.EncodeToString(hasher.Sum(nil))
	signatured := fmt.Sprintf("%s|%s", tariffKey, hashedTariffKey)
	var params map[string]string
	if builder.queryParameters.Flags.UseUnisearch() {
		params = map[string]string{
			"partner":            variant.PartnerCode,
			"qid":                variant.Qid,
			"forward":            variant.Forward.RedirectKey,
			"unisearchTariffKey": signatured,
			"tariff_sign":        SerializeTariff(&variant.Tariff, variant.CreatedAt, builder.settings.TDApiRedirectSecretKey),
		}
	} else {
		params = map[string]string{
			"partner":         variant.PartnerCode,
			"qid":             variant.Qid,
			"forward":         variant.Forward.RedirectKey,
			"wizardTariffKey": signatured,
			"tariff_sign":     SerializeTariff(&variant.Tariff, variant.CreatedAt, builder.settings.TDApiRedirectSecretKey),
		}
	}

	if builder.queryParameters.Flags.EnablePrimaryPartnerFreeRedirect() {
		if aviaDynamic := builder.queryParameters.AviaDynamic; aviaDynamic != nil {
			if primaryPartnerCode := aviaDynamic.PrimaryPartnerCode; primaryPartnerCode != nil {
				selectedPartners, _ := aviaDynamic.Filters.Partners()
				if len(selectedPartners) == 1 && selectedPartners.Contains(*primaryPartnerCode) {
					params["aviaBrand"] = *primaryPartnerCode
				}
			}
		} else {
			if builder.queryParameters.PartnerCode != nil {
				params["aviaBrand"] = *builder.queryParameters.PartnerCode
			}
		}
	}

	if len(variant.Backward.Flights) > 0 {
		params["backward"] = variant.Backward.RedirectKey
	}
	useTravelPortal := builder.queryParameters.UseTravelPortal()
	urlTemplate := "%s/order/redirect/"
	if useTravelPortal {
		urlTemplate = "%s/avia/redirect/"
	}
	return NewLanding(fmt.Sprintf(urlTemplate, builder.baseRedirect(useTravelPortal))).WithParams(params)
}

func formatFlight(flight *responses.PPFlight) string {
	return fmt.Sprintf("%s.%s", flight.Number, flight.DepartsAt[:16])
}

func formatTrip(trip *responses.Trip) *string {
	if helpers.IsNil(trip) || len(trip.Flights) == 0 {
		return nil
	}
	var formattedSegments []string
	for _, flight := range trip.Flights {
		formattedSegments = append(formattedSegments, formatFlight(flight))
	}
	result := strings.Join(formattedSegments, ",")
	return &result
}

func (builder FrontLandingBuilder) OrderURL(
	from, to *models.Settlement,
	variant *responses.Variant,
	useTravelPortal consts.TravelPortalUsage,
) *Landing {
	params := map[string]string{
		"when":   variant.Forward.Flights[0].DepartsAt[:10],
		"fromId": from.GetPointKey(),
		"toId":   to.GetPointKey(),
	}
	if forward := formatTrip(variant.Forward); !helpers.IsNil(forward) {
		params["forward"] = *forward
	}
	if backward := formatTrip(variant.Backward); !helpers.IsNil(backward) {
		params["backward"] = *backward
		params["return_date"] = variant.Backward.Flights[0].DepartsAt[:10]
	}
	if useTravelPortal {
		return NewLanding(fmt.Sprintf("%s/avia/order/", builder.base(consts.UseTravelPortal))).WithParams(params)
	}
	return NewLanding(fmt.Sprintf("%s/order/", builder.base(consts.UseTravelPortal))).WithParams(params)
}

func (builder FrontLandingBuilder) Routes(
	fromSettlement, toSettlement *models.Settlement,
	lang models.Lang,
	passengers *dynamic.Passengers,
	toCrimea bool,
) *Landing {
	toKey := "krym"
	toApp := "krym"
	if !toCrimea {
		toKey = getSettlementKey(toSettlement)
		toSettlementTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(toSettlement.NewLTitleID, lang, consts.CaseNominative)
		toApp = getAppendix(toSettlementTitle)
	}

	fromSettlementTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(fromSettlement.NewLTitleID, lang, consts.CaseNominative)
	useTravelPortal := builder.queryParameters.UseTravelPortal()
	landing := NewLanding(
		fmt.Sprintf(
			"%s/routes/%s/%s/%s-%s",
			builder.base(useTravelPortal),
			getSettlementKey(fromSettlement),
			toKey,
			getAppendix(fromSettlementTitle),
			toApp,
		),
	)

	if helpers.IsNil(passengers) {
		return landing
	}

	return landing.WithParams(
		map[string]string{
			"adults_seats":   strconv.Itoa(passengers.Adults),
			"children_seats": strconv.Itoa(passengers.Children),
			"infants_seats":  strconv.Itoa(passengers.Infants),
		},
	)
}

func (builder FrontLandingBuilder) FlightRoutes(
	airportFromCode, airportToCode string,
	fromPoint, toPoint models.Point,
	lang models.Lang,
) *Landing {
	toPointTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(toPoint.GetTitleID(), lang, consts.CaseNominative)
	fromSettlementTitle, _ := builder.translatedTitleRepository.GetTitleTranslation(fromPoint.GetTitleID(), lang, consts.CaseNominative)
	return NewLanding(
		fmt.Sprintf(
			"%s/routes/%s/%s/%s-%s",
			builder.base(consts.DontUseTravelPortal),
			getPointKeyForFlight(airportFromCode, fromPoint),
			getPointKeyForFlight(airportToCode, toPoint),
			getAppendix(fromSettlementTitle),
			getAppendix(toPointTitle),
		),
	)
}

func (builder FrontLandingBuilder) Flight(number string, date *time.Time) *Landing {
	params := make(stringMap)
	if date != nil {
		params["when"] = date.Format("2006-01-02")
	}

	number = strings.ReplaceAll(number, " ", "-")
	useTravelPortal := builder.queryParameters.UseTravelPortal()
	urlTemplate := "%s/flights/%s"
	if useTravelPortal {
		urlTemplate = "%s/avia/flights/%s/"
	}
	URL := fmt.Sprintf(urlTemplate, builder.base(useTravelPortal), number)
	return NewLanding(URL).WithParams(params)
}

func (builder FrontLandingBuilder) AirlineFromSettlement(company *models.Company, settlement *models.Settlement) *Landing {
	URL := fmt.Sprintf("%s/airline/%s", builder.base(consts.DontUseTravelPortal), company.Slug)
	return NewLanding(URL).WithParam("fromId", settlement.GetPointKey())
}

func (builder FrontLandingBuilder) baseRedirect(useTravelPortal consts.TravelPortalUsage) string {
	if useTravelPortal {
		urls := builder.settings.TravelPortalURLs
		if builder.queryParameters.Flags.UseXRedirect() {
			urls = builder.settings.XRedirectURLs
		}
		if baseURL, ok := urls[builder.queryParameters.NationalVersion]; ok {
			return baseURL
		}
	}
	return builder.settings.FrontendURLs[builder.queryParameters.NationalVersion]
}

func (builder FrontLandingBuilder) base(useTravelPortal consts.TravelPortalUsage) string {
	if useTravelPortal {
		if baseURL, ok := builder.settings.TravelPortalURLs[builder.queryParameters.NationalVersion]; ok {
			return baseURL
		}
	}
	return builder.settings.FrontendURLs[builder.queryParameters.NationalVersion]
}

func (builder FrontLandingBuilder) City(settlementTitle, settlementIATA, settlementKey string) *Landing {
	if builder.queryParameters.UseTravelPortal() {
		return NewLanding(fmt.Sprintf("%s/avia", builder.base(consts.UseTravelPortal))).WithParam("fromId", settlementKey)
	}
	key := settlementIATA
	if key == "" {
		key = settlementKey
	}
	key = strings.ToLower(key)
	return NewLanding(fmt.Sprintf("%s/city/%s/-%s", builder.base(consts.DontUseTravelPortal), key, getAppendix(settlementTitle)))
}

func getSettlementKey(point *models.Settlement) string {
	key := point.GetIata()
	if key == "" {
		key = point.GetPointKey()
	}
	return strings.ToLower(key)
}

func getPointKeyForFlight(airportCode string, point models.Point) string {
	key := airportCode
	if point.GetPointType() == models.PointTypeSettlement {
		key = point.(*models.Settlement).IataCode
	}
	if key == "" {
		key = point.GetPointKey()
	}
	return strings.ToLower(key)
}

func getAppendix(title string) string {
	title = strings.ToLower(title)
	title = strings.ReplaceAll(title, "'", "")
	return url.PathEscape(translations.Unidecode(title))
}
