package landings

import (
	"fmt"
	"net/url"
	"strings"
)

type (
	stringMap map[string]string

	Landing struct {
		base    string
		params  stringMap
		anchors stringMap
	}
)

func NewLanding(base string) *Landing {
	return &Landing{
		base:    base,
		params:  make(stringMap),
		anchors: make(stringMap),
	}
}

func (landing Landing) WithParams(params stringMap) *Landing {
	return &Landing{
		base:    landing.base,
		params:  mergeMap(landing.params, params),
		anchors: copyMap(landing.anchors),
	}
}

func (landing Landing) WithParam(key, value string) *Landing {
	return landing.WithParams(map[string]string{key: value})
}

func (landing Landing) WithAnchors(anchors stringMap) *Landing {
	return &Landing{
		base:    landing.base,
		params:  copyMap(landing.params),
		anchors: mergeMap(landing.anchors, anchors),
	}
}

func (landing Landing) WithAnchor(key, value string) *Landing {
	return landing.WithAnchors(map[string]string{key: value})
}

func (landing Landing) AsString() string {
	URL, _ := url.Parse(landing.base)
	query := URL.Query()

	for key, value := range landing.params {
		query.Set(key, value)
	}
	URL.RawQuery = query.Encode()

	if len(landing.anchors) != 0 {
		fragments := make([]string, 0, len(landing.anchors))
		for key, value := range landing.anchors {
			fragments = append(fragments, fmt.Sprintf("%s=%s", key, value))
		}
		URL.Fragment = strings.Join(fragments, "&")
	}

	return URL.String()
}

func mergeMap(left, right stringMap) stringMap {
	result := copyMap(left)

	for key, value := range right {
		result[key] = value
	}
	return result
}

func copyMap(mapForCopy stringMap) stringMap {
	result := make(stringMap)

	for key, value := range mapForCopy {
		result[key] = value
	}

	return result
}
