package landings

import (
	"crypto/md5"
	"encoding/hex"
	"fmt"
	"strconv"
	"strings"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
)

func SerializeTariff(tariff *responses.Tariff, createdAt int32, secretKey string) string {
	if secretKey == "" || tariff.Currency == "" {
		return ""
	}
	hash := getHash(int(tariff.Price), tariff.Currency, secretKey)
	return fmt.Sprintf("%d|%s|%s|%d", int(tariff.Price), tariff.Currency, hash, createdAt)
}

type tariffDeserializationError error

func newTariffDeserializationError(serializedTariff string) tariffDeserializationError {
	return fmt.Errorf("bad serialized tariff format: %s", serializedTariff)
}

func DeserializeTariff(serializedTariff, secretKey string) (tariff *responses.Tariff, createdAt int32, err error) {
	parts := strings.Split(serializedTariff, "|")
	if len(parts) < 4 {
		return nil, 0, newTariffDeserializationError(serializedTariff)
	}
	price, err := strconv.Atoi(parts[0])
	if err != nil {
		return nil, 0, newTariffDeserializationError(serializedTariff)
	}
	currency := parts[1]
	if getHash(price, currency, secretKey) != parts[2] {
		return nil, 0, newTariffDeserializationError(serializedTariff)
	}

	createdAtValue, err := strconv.Atoi(parts[3])
	if err != nil {
		return nil, 0, fmt.Errorf("bad serialized tariff format: %s", serializedTariff)
	}
	return &responses.Tariff{
		Currency: currency,
		Price:    float64(price),
	}, int32(createdAtValue), nil
}

func getHash(price int, currency, secretKey string) string {
	hasher := md5.New()
	hasher.Write([]byte(fmt.Sprintf("%d|%s%s", price, currency, secretKey)))
	return hex.EncodeToString(hasher.Sum(nil))
}
