package mappers

import (
	"context"
	"fmt"
	"math"
	"sync"
	"time"

	"github.com/opentracing/opentracing-go"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/filtering"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories/ydb"
	wizardProto "a.yandex-team.ru/travel/proto/avia/wizard"
)

type WizardToSearchResultMapper struct {
	stationRepository         repositories.Station
	partnerRepository         repositories.Partner
	aviaCompanyRepository     repositories.AviaCompany
	companyTariffRepository   repositories.CompanyTariff
	logger                    log.Logger
	companyRepository         repositories.Company
	translatedTitleRepository repositories.TranslatedTitle
	locationRepository        repositories.CachedLocation
	checkedBaggageRepository  repositories.CachedCheckedBaggage
}

func NewWizardToSearchResultMapper(
	logger log.Logger,
	stationRepository repositories.Station,
	partnerRepository repositories.Partner,
	aviaCompanyRepository repositories.AviaCompany,
	companyRepository repositories.Company,
	translatedTitleRepository repositories.TranslatedTitle,
	companyTariffRepository repositories.CompanyTariff,
	locationRepository repositories.CachedLocation,
	checkedBaggageRepository repositories.CachedCheckedBaggage,
) *WizardToSearchResultMapper {
	return &WizardToSearchResultMapper{
		logger:                    logger,
		stationRepository:         stationRepository,
		partnerRepository:         partnerRepository,
		aviaCompanyRepository:     aviaCompanyRepository,
		companyTariffRepository:   companyTariffRepository,
		companyRepository:         companyRepository,
		translatedTitleRepository: translatedTitleRepository,
		locationRepository:        locationRepository,
		checkedBaggageRepository:  checkedBaggageRepository,
	}
}

func (mapper *WizardToSearchResultMapper) Map(
	result *ydb.WizardSearchResult,
	queryParameters *parameters.QueryParameters,
	ctx context.Context,
) *results.SearchResult {
	span, _ := opentracing.StartSpanFromContext(ctx, "Mapping YDB results")
	defer span.Finish()
	if result == nil {
		return mapper.buildCacheMissResult(ctx, queryParameters)
	}
	dateForward := mapper.mapDate(result.DateForward)
	var dateBackward *time.Time
	if result.DateBackward > 0 {
		mappedDate := mapper.mapDate(result.DateBackward)
		dateBackward = &mappedDate
	}
	withBaggage := false
	if !helpers.IsNil(queryParameters.AviaDynamic) {
		if withBaggageValue, ok := queryParameters.Filters().WithBaggage(); ok && withBaggageValue != nil && *withBaggageValue {
			withBaggage = true
		}
	}
	utcNow := time.Now().UTC()
	faresProtoFilter := filtering.NewFaresProtoFilter(mapper.logger, result.SearchResult.Value.Flights, queryParameters.JobID).
		ApplyUserFilters(mapper.locationRepository, queryParameters.Filters()).
		DisabledPartner(mapper.partnerRepository, queryParameters.NationalVersion).
		IncorrectCompanyIDs(mapper.companyRepository).
		NoStationTranslations(mapper.stationRepository, mapper.translatedTitleRepository, queryParameters.Lang).
		DepartsSoon(utcNow, mapper.locationRepository).
		ShortStopover(mapper.locationRepository).
		PromoExpired(utcNow)

	fareGenerator := results.NewFareGenerator(
		result.SearchResult.Value.Fares,
		result.SearchResult.Value.Flights,
		*result.SearchResult.Value.Qid,
		withBaggage,
		mapper.mapFare,
		faresProtoFilter.IsGoodFare,
	)

	totalFaresCount := len(result.SearchResult.Value.Fares)
	if queryParameters.Flags.EnablePartnerCache() {
		if partners, ok := queryParameters.Filters().Partners(); (ok && len(partners) > 0) || (queryParameters.PartnerCode != nil && *queryParameters.PartnerCode != "") {
			totalFaresCount = int(*result.SearchResult.Value.OffersCount)
		}
	}

	return &results.SearchResult{
		MinPrice:        float32(result.MinPrice) / 100,
		DateForward:     &dateForward,
		DateBackward:    dateBackward,
		FareGenerator:   fareGenerator,
		TotalFaresCount: totalFaresCount,
		Filters:         mapper.buildFilters(&result.FilterState.Value, queryParameters.AviaDynamic),
		PollingStatus:   results.NewPollingStatusFromProto(result.SearchResult.Value.PollingStatus),
		Version:         *result.SearchResult.Value.Version,
	}
}

func (mapper *WizardToSearchResultMapper) mapDate(date uint32) time.Time {
	return time.Unix(int64(consts.SecondsInDay*date), 0)
}

func (mapper *WizardToSearchResultMapper) mapFares(
	faresProto []*wizardProto.Fare,
	flightProtos map[string]*wizardProto.Flight,
	qid string,
	withBaggage bool,
	ctx context.Context,
) results.Fares {
	span, _ := opentracing.StartSpanFromContext(ctx, "Mapping fares")
	defer span.Finish()
	if faresProto == nil {
		return nil
	}
	fares := make([]*results.Fare, 0, len(faresProto))
	for _, f := range faresProto {
		fare, err := mapper.mapFare(f, flightProtos, qid, withBaggage)
		if err != nil {
			mapper.logger.Warn(fmt.Sprintf("couldn't decode fare: %+v", err))
			continue
		}
		fares = append(fares, fare)
	}
	return fares
}

func (mapper *WizardToSearchResultMapper) mapFare(
	fareProto *wizardProto.Fare,
	flightProtos map[string]*wizardProto.Flight,
	qid string,
	withBaggage bool,
) (*results.Fare, error) {
	var forwardTrip results.Trip
	var backwardTrip results.Trip
	errorGroup, _ := errgroup.WithContext(context.Background())
	errorGroup.Go(func() error {
		var err error
		forwardTrip, err = mapper.mapTrip(fareProto.Route.Forward, flightProtos)
		return err
	})
	errorGroup.Go(func() error {
		var err error
		backwardTrip, err = mapper.mapTrip(fareProto.Route.Backward, flightProtos)
		return err
	})
	var baggage *results.Baggage
	errorGroup.Go(func() error {
		var err error
		baggage, err = mapper.mapBaggage(fareProto, flightProtos, withBaggage)
		return err
	})
	if err := errorGroup.Wait(); err != nil {
		return nil, err
	}
	partner, found := mapper.partnerRepository.GetPartnerByCode(*fareProto.Partner)
	if !found {
		return nil, fmt.Errorf("unknown partner %s in fare", *fareProto.Partner)
	}

	tariff, err := mapper.mapTariff(fareProto.Tariff)
	if err != nil {
		return nil, err
	}
	return &results.Fare{
		ForwardTrip:     forwardTrip,
		BackwardTrip:    backwardTrip,
		PartnerCode:     *fareProto.Partner,
		Tariff:          *tariff,
		Qid:             qid,
		FromAviaCompany: partner.IsAviacompany,
		ExpiresAt:       *fareProto.ExpireAt,
		CreatedAt:       *fareProto.CreatedAt,
		Popularity:      *fareProto.Popularity,
		Baggage:         baggage,
		IsPromo:         fareProto.Promo != nil && fareProto.Promo.Code != nil && *fareProto.Promo.Code != "",
	}, nil
}

func (mapper *WizardToSearchResultMapper) mapTrip(
	flightKeys []string,
	flights map[string]*wizardProto.Flight,
) (results.Trip, error) {
	trip := make(results.Trip, 0, len(flightKeys))

	for _, key := range flightKeys {
		flight := flights[key]
		waitGroup := sync.WaitGroup{}
		waitGroup.Add(2)
		var departureTime time.Time
		var arrivalTime time.Time
		go func() {
			defer waitGroup.Done()
			departureTime = mapper.parseFlightTime(flight.Departure)
		}()
		go func() {
			defer waitGroup.Done()
			arrivalTime = mapper.parseFlightTime(flight.Arrival)
		}()
		stationFrom, _ := mapper.stationRepository.GetByID(int(*flight.FromId))
		stationTo, _ := mapper.stationRepository.GetByID(int(*flight.ToId))
		waitGroup.Wait()
		trip = append(trip, &results.Segment{
			Number:      *flight.Number,
			Departure:   departureTime,
			Arrival:     arrivalTime,
			CompanyID:   int(*flight.Company),
			StationFrom: stationFrom,
			StationTo:   stationTo,
		})
	}
	return trip, nil
}

func (mapper *WizardToSearchResultMapper) mapTariff(tariffProto *wizardProto.Tariff) (*results.Tariff, error) {
	if tariffProto == nil {
		return nil, fmt.Errorf("no tariff in fare")
	}
	return &results.Tariff{
		Currency: *tariffProto.Currency,
		Value:    *tariffProto.Value,
	}, nil
}

func (mapper *WizardToSearchResultMapper) parseFlightTime(timestamp *wizardProto.LocalTimestamp) time.Time {
	location, _ := mapper.locationRepository.LoadLocation(*timestamp.Tzname)
	parsed, _ := helpers.ParseTimeInLocation(*timestamp.Local, location)
	return parsed
}

func (mapper *WizardToSearchResultMapper) mapBaggage(
	fareProto *wizardProto.Fare,
	flightProtos map[string]*wizardProto.Flight,
	withBaggage bool,
) (*results.Baggage, error) {
	if fareProto == nil {
		return nil, nil
	}
	companyIDs := mapper.getCompanyIDs(fareProto.Route.Forward, fareProto.Route.Backward, flightProtos)
	var checkedBaggage *results.Checked
	var carryOnBaggage *results.CarryOn
	errorGroup, _ := errgroup.WithContext(context.Background())
	errorGroup.Go(func() error {
		var err error
		checkedBaggage, err = mapper.parseChecked(fareProto.Baggage, companyIDs)
		return err
	})
	errorGroup.Go(func() error {
		carryOnBaggage = mapper.parseFareCarryOn(companyIDs)
		return nil
	})

	if err := errorGroup.Wait(); err != nil {
		return nil, err
	}
	return &results.Baggage{
		CarryOn: carryOnBaggage,
		Checked: checkedBaggage,
	}, nil
}

func (mapper *WizardToSearchResultMapper) getCompanyIDs(
	forward, backward []string,
	flightProtos map[string]*wizardProto.Flight,
) containers.SetOfInt {
	companyIDs := make(containers.SetOfInt, len(forward)+len(backward))
	for _, key := range forward {
		flight := flightProtos[key]
		companyIDs.Add(int(*flight.Company))
	}
	for _, key := range backward {
		flight := flightProtos[key]
		companyIDs.Add(int(*flight.Company))
	}
	return companyIDs
}

func (mapper *WizardToSearchResultMapper) parseFareCarryOn(companyIDs containers.SetOfInt) *results.CarryOn {
	minLength := math.MaxInt32
	minWidth := math.MaxInt32
	minHeight := math.MaxInt32
	minSum := math.MaxInt32

	for id := range companyIDs {
		aviaCompany, found := mapper.aviaCompanyRepository.GetByID(id)
		if !found {
			return &results.CarryOn{}
		}
		if aviaCompany.CarryOnHeight == 0 || aviaCompany.CarryOnLength == 0 ||
			aviaCompany.CarryOnWidth == 0 || aviaCompany.CarryOnDimensionsSum == 0 {
			continue
		}

		if aviaCompany.CarryOnLength < minLength {
			minLength = aviaCompany.CarryOnLength
		}
		if aviaCompany.CarryOnWidth < minLength {
			minWidth = aviaCompany.CarryOnWidth
		}
		if aviaCompany.CarryOnHeight < minLength {
			minHeight = aviaCompany.CarryOnHeight
		}
	}
	if minLength != math.MaxInt32 && minWidth != math.MaxInt32 && minHeight != math.MaxInt32 {
		if minSum > minHeight+minWidth+minLength {
			minSum = minHeight + minWidth + minLength
		}
	}
	var included *bool
	var minCarryOnNorm *float64

	for companyID := range companyIDs {
		companyTariffs, found := mapper.companyTariffRepository.GetTariffsByCompanyID(companyID)
		if !found {
			continue
		}
		for _, tariff := range companyTariffs {
			if tariff.CarryOnNorm != 0 && (minCarryOnNorm == nil || float32(*minCarryOnNorm) > tariff.CarryOnNorm) {
				minCarryOnNorm = ptr.Float64(float64(tariff.CarryOnNorm))
			}
			if included == nil && tariff.CarryOn != 0 || tariff.CarryOn == 0 {
				included = ptr.Bool(tariff.CarryOn == 1)
			}
		}
	}

	if included == nil || !*included {
		return &results.CarryOn{Included: included}
	}

	return &results.CarryOn{
		Weight:        minCarryOnNorm,
		DimensionsSum: &minSum,
		Height:        &minHeight,
		Width:         &minWidth,
		Length:        &minLength,
		Included:      included,
	}
}

func (mapper *WizardToSearchResultMapper) parseChecked(baggage *wizardProto.Baggage, companyIDs containers.SetOfInt) (*results.Checked, error) {
	if baggage == nil {
		return &results.Checked{Dimensions: &results.Dimensions{}}, nil
	}
	baggages := containers.NewSetOfStringFromSlices(baggage.Forward, baggage.Backward)

	if len(baggages) == 0 || baggages.Contains("") {
		return &results.Checked{
			Dimensions: mapper.getWorstBaggageDimensions(companyIDs),
		}, nil
	}
	overallIncluded := ptr.Int(1)
	overallWeight := ptr.Int(100)
	overallPieces := ptr.Int(100)

	min := func(a, b *int) *int {
		if a == nil || b == nil {
			return nil
		}
		if *a < *b {
			return a
		}
		return b
	}
	for code := range baggages {
		checkedBaggage, err := mapper.checkedBaggageRepository.ParseBaggageCode(code)
		if err != nil {
			return &results.Checked{}, err
		}
		overallIncluded = min(overallIncluded, checkedBaggage.Included)
		overallWeight = min(overallWeight, checkedBaggage.Weight)
		overallPieces = min(overallPieces, checkedBaggage.Pieces)
	}

	checked := results.Checked{
		Included:   new(bool),
		Dimensions: mapper.getWorstBaggageDimensions(companyIDs),
		Weight:     overallWeight,
		Pieces:     overallPieces,
	}
	if overallIncluded != nil {
		if *overallIncluded == 1 {
			*checked.Included = true
		} else {
			*checked.Included = false
		}
	}
	return &checked, nil
}

func (mapper *WizardToSearchResultMapper) getWorstBaggageDimensions(companyIDs containers.SetOfInt) *results.Dimensions {
	var dimensions *results.Dimensions
	for companyID := range companyIDs {
		aviaCompany, found := mapper.aviaCompanyRepository.GetByID(companyID)
		if !found {
			continue
		}

		if dimensions == nil {
			dimensions = &results.Dimensions{
				Height: aviaCompany.BaggageHeight,
				Length: aviaCompany.BaggageLength,
				Width:  aviaCompany.BaggageHeight,
				Sum:    &aviaCompany.BaggageDimensionsSum,
			}
		} else {
			dimensions = dimensions.Merge(aviaCompany)
		}
	}
	return dimensions
}

func (mapper *WizardToSearchResultMapper) buildCacheMissResult(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
) *results.SearchResult {
	departureDate := queryParameters.DepartureDate
	if departureDate == nil {
		departureDate = helpers.PredictDate(queryParameters.UserTime)
	}
	props.SetSearchProp(ctx, "cache_miss", "1")
	return &results.SearchResult{
		MinPrice:        0,
		DateForward:     departureDate,
		DateBackward:    nil,
		FareGenerator:   results.EmptyFareGenerator,
		Filters:         nil,
		PollingStatus:   nil,
		Version:         0,
		TotalFaresCount: 0,
		IsCacheMiss:     true,
	}
}

func (mapper *WizardToSearchResultMapper) buildFilters(
	filterState *results.FilterState,
	aviaDynamic *dynamic.AviaDynamic,
) *results.FilterState {
	if helpers.IsNil(aviaDynamic) {
		return filterState
	}
	filters := aviaDynamic.Filters
	airlines, airlinesOk := filters.Airlines()
	partners, partnersOk := filters.Partners()
	if !airlinesOk && !partnersOk {
		return filterState
	}
	if airlinesOk {
		fillAirlines(filterState, airlines)
	}
	if partnersOk {
		fillPartners(filterState, partners)
	}
	return filterState
}

func fillAirlines(filterState *results.FilterState, airlines containers.SetOfInt) {
	existingAirlines := containers.SetOfInt{}
	if helpers.IsNil(filterState.Airlines) {
		filterState.Airlines = &containers.SetOfInt{}
	}
	for airlineID := range *filterState.Airlines {
		existingAirlines.Add(airlineID)
	}
	airlinesToAdd := []int{}
	if airlines != nil {
		airlinesToAdd = airlines.ToSlice()
	}
	for _, airlineID := range airlinesToAdd {
		filterState.Airlines.Add(airlineID)
	}
}

func fillPartners(filterState *results.FilterState, partners containers.SetOfString) {
	existingPartners := containers.SetOfString{}
	if helpers.IsNil(filterState.Partners) {
		filterState.Partners = &containers.SetOfString{}
	}
	for partnerCode := range *filterState.Partners {
		existingPartners.Add(partnerCode)
	}
	partnersToAdd := []string{}
	if partners != nil {
		partnersToAdd = partners.ToSlice()
	}
	for _, partnerCode := range partnersToAdd {
		filterState.Partners.Add(partnerCode)
	}
}
