package parameters

import (
	"net/url"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flags"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
)

type QueryParameters struct {
	MainReqID        string
	ReqID            string
	GeoID            int
	FromID           *string
	ToID             *string
	FromText         string
	ToText           string
	FromGeoID        *int
	ToGeoID          *int
	PointFrom        string
	PointTo          string
	DepartureDate    *time.Time
	FlightDate       *time.Time
	ReturnDate       *time.Time
	CompanyID        *int
	PartnerCode      *string
	DirectFlight     bool
	FlightNumber     *string
	NationalVersion  string
	Tld              string
	Lang             models.Lang
	SerpUUID         string
	UserRequest      string
	Device           string
	YandexUID        string
	PassportID       string
	ICookieDecrypted string
	Flags            flags.Flags
	AviaDynamic      *dynamic.AviaDynamic
	Dev              bool
	JobID            string
	RawArgs          url.Values
	Company          *models.Company
	filters          dynamic.Filters
	UserTime         time.Time
	QueryHasDate     bool
}

func (queryParameters *QueryParameters) IsDynamic() bool {
	return queryParameters.AviaDynamic != nil
}

func (queryParameters *QueryParameters) IsTouchDevice() bool {
	return queryParameters.Device == "touch"
}

func (queryParameters *QueryParameters) Passengers() dynamic.Passengers {
	if queryParameters.IsDynamic() {
		return queryParameters.AviaDynamic.Passengers
	}
	return dynamic.Passengers{Adults: 1}
}

func (queryParameters *QueryParameters) Context() dynamic.Context {
	if helpers.IsNil(queryParameters.AviaDynamic) {
		return dynamic.Context{}
	}
	contextCopy := map[string]interface{}{}
	for k, v := range queryParameters.AviaDynamic.Context {
		contextCopy[k] = v
	}
	if val, ok := contextCopy["utm-medium"]; ok {
		contextCopy["utm_medium"] = val
		delete(contextCopy, "utm-medium")
	}
	return contextCopy
}

func (queryParameters *QueryParameters) Filters() dynamic.Filters {
	if !helpers.IsNil(queryParameters.filters) {
		return queryParameters.filters
	}

	if !helpers.IsNil(queryParameters.AviaDynamic) {
		_, airlinesOk := queryParameters.AviaDynamic.Filters.Airlines()
		_, partnersOk := queryParameters.AviaDynamic.Filters.Partners()
		if airlinesOk || partnersOk {
			queryParameters.filters = queryParameters.AviaDynamic.Filters
			return queryParameters.AviaDynamic.Filters
		}
	}

	filters := dynamic.Filters{}
	if !helpers.IsNil(queryParameters.AviaDynamic) {
		filters = queryParameters.AviaDynamic.Filters
	} else {
		if queryParameters.PartnerCode != nil && *queryParameters.PartnerCode != "" {
			filters["partners"] = containers.NewSetOfString(*queryParameters.PartnerCode)
		}
	}
	if queryParameters.Company != nil {
		filters["airlines"] = containers.NewSetOfInt(*queryParameters.CompanyID)
	}
	queryParameters.filters = filters
	return filters
}

func (queryParameters *QueryParameters) UseTravelPortal() consts.TravelPortalUsage {
	return queryParameters.NationalVersion == "ru" && queryParameters.Tld == "ru"
}

func Compare(first *QueryParameters, second *QueryParameters) bool {
	return first.MainReqID == second.MainReqID &&
		first.ReqID == second.ReqID &&
		first.GeoID == second.GeoID &&
		optionalStringsAreEqual(first.FromID, second.FromID) &&
		optionalStringsAreEqual(first.ToID, second.ToID) &&
		first.FromText == second.FromText &&
		first.ToText == second.ToText &&
		*first.FromGeoID == *second.FromGeoID &&
		*first.ToGeoID == *second.ToGeoID &&
		first.PointFrom == second.PointFrom &&
		first.PointTo == second.PointTo &&
		first.DepartureDate == second.DepartureDate &&
		first.FlightDate == second.FlightDate &&
		first.ReturnDate == second.ReturnDate &&
		optionalIntsAreEqual(first.CompanyID, second.CompanyID) &&
		optionalStringsAreEqual(first.FlightNumber, second.FlightNumber) &&
		first.NationalVersion == second.NationalVersion &&
		first.Tld == second.Tld &&
		first.Lang == second.Lang &&
		first.SerpUUID == second.SerpUUID &&
		first.Device == second.Device &&
		first.YandexUID == second.YandexUID &&
		first.ICookieDecrypted == second.ICookieDecrypted &&
		first.Context().Equals(second.Context()) &&
		first.Flags.Equals(&second.Flags) &&
		first.Dev == second.Dev &&
		first.QueryHasDate == second.QueryHasDate
}

func optionalStringsAreEqual(first, second *string) bool {
	return helpers.IsNil(first) == helpers.IsNil(second) && (helpers.IsNil(first) || *first == *second)
}

func optionalIntsAreEqual(first, second *int) bool {
	return helpers.IsNil(first) == helpers.IsNil(second) && (helpers.IsNil(first) || *first == *second)
}
