package parameters

import (
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"a.yandex-team.ru/travel/avia/library/go/utils"
	wizardContext "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/context"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flags"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/checkers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type IQueryParametersParser interface {
	Parse(request *http.Request) (*QueryParameters, error)
}

type QueryParametersParser struct {
	aviaDynamicParser dynamic.IAviaDynamicParser
	flagParser        flags.Parser
	checker           checkers.QueryParametersChecker
	companyRepository repositories.Company
}

func NewQueryParametersParser(
	aviaDynamicParser dynamic.IAviaDynamicParser,
	flagParser flags.Parser,
	checker checkers.QueryParametersChecker,
	companyRepository repositories.Company,
) *QueryParametersParser {
	return &QueryParametersParser{
		aviaDynamicParser: aviaDynamicParser,
		flagParser:        flagParser,
		checker:           checker,
		companyRepository: companyRepository,
	}
}

func (parser *QueryParametersParser) Parse(request *http.Request) (_ *QueryParameters, err error) {
	queryParameters := utils.HTTP.ParseQueryParameters(request)

	err = parser.checker.CheckParameters(request)
	if err != nil {
		return nil, err
	}

	mainReqID := queryParameters.Get("main_reqid")
	reqID := queryParameters.Get("reqid")
	geoID, err := strconv.Atoi(queryParameters.Get("geo_id"))
	if err != nil {
		geoID = -1
	}

	pointFrom := queryParameters.Get("point_from")
	pointTo := queryParameters.Get("point_to")
	fromID := getOptionalString(queryParameters.Get("from_id"))
	toID := getOptionalString(queryParameters.Get("to_id"))
	fromText := queryParameters.Get("from_text")
	toText := queryParameters.Get("to_text")
	flightNumber := getOptionalString(queryParameters.Get("flight_number"))
	fromGeoID := getOptionalInt(queryParameters.Get("from_geo_id"))
	toGeoID := getOptionalInt(queryParameters.Get("to_geo_id"))

	departureDate, returnDate, flightDate, err := parseDates(queryParameters)
	if err != nil {
		return nil, err
	}

	lang := queryParameters.Get("lang")
	tld := queryParameters.Get("tld")
	serpUUID := queryParameters.Get("serp_uuid")
	userRequest := queryParameters.Get("user_request")
	device := queryParameters.Get("device")
	yandexUID := queryParameters.Get("yandexuid")
	passportID := queryParameters.Get("passport_id")
	icookieDecrypted := queryParameters.Get("icookie_decrypted")

	aviaDynamic, err := parser.aviaDynamicParser.Parse(queryParameters.Get("avia_dynamic"))
	if err != nil {
		return nil, err
	}

	parsedFlags, err := parser.flagParser.Parse(queryParameters.Get("content_flags"))
	if err != nil {
		return nil, err
	}

	companyID := getOptionalInt(queryParameters.Get("company"))
	if aviaDynamic != nil {
		if company, ok := aviaDynamic.Context.Company(); ok && companyID == nil {
			companyID = company
		}
	}
	var company *models.Company
	if companyID != nil {
		if c, ok := parser.companyRepository.GetByID(*companyID); ok {
			company = c
		}
	}

	partnerCode := getOptionalString(queryParameters.Get("partner_code"))

	directFlight := queryParameters.Get("direct_flight") == "1"
	dev := queryParameters.Get("dev") == "1"
	nationalVersion := getNationalVersionByTld(tld, parsedFlags)

	jobID := wizardContext.GetJobID(request.Context())

	return &QueryParameters{
		MainReqID: mainReqID,
		ReqID:     reqID,
		JobID:     jobID,

		GeoID:     geoID,
		FromID:    fromID,
		ToID:      toID,
		FromText:  fromText,
		ToText:    toText,
		FromGeoID: fromGeoID,
		ToGeoID:   toGeoID,
		PointFrom: pointFrom,
		PointTo:   pointTo,

		DepartureDate: departureDate,
		ReturnDate:    returnDate,
		FlightDate:    flightDate,

		CompanyID:        companyID,
		Company:          company,
		PartnerCode:      partnerCode,
		DirectFlight:     directFlight,
		FlightNumber:     flightNumber,
		NationalVersion:  nationalVersion,
		Tld:              tld,
		Lang:             getLangModel(lang),
		SerpUUID:         serpUUID,
		UserRequest:      userRequest,
		Device:           device,
		YandexUID:        yandexUID,
		PassportID:       passportID,
		ICookieDecrypted: icookieDecrypted,
		Flags:            parsedFlags,
		AviaDynamic:      aviaDynamic,
		RawArgs:          queryParameters,
		Dev:              dev,
		QueryHasDate:     queryParameters.Get("query_has_date") == "1",
	}, nil
}

func parseDates(queryParameters url.Values) (*time.Time, *time.Time, *time.Time, error) {
	departureDate, err := tryParseDate(queryParameters.Get("departure_date"))
	if err != nil {
		return nil, nil, nil, domain.NewWizardError(
			fmt.Sprintf(
				"catch error on parse departure_date: %v",
				queryParameters.Get("departure_date"),
			), "bad_date",
		)
	}
	returnDate, err := tryParseDate(queryParameters.Get("return_date"))
	if err != nil {
		return nil, nil, nil, domain.NewWizardError(
			fmt.Sprintf("catch error on parse return_date: %v", queryParameters.Get("return_date")),
			"bad_date",
		)
	}
	flightDate, err := tryParseDate(queryParameters.Get("flight_date"))
	if err != nil {
		return nil, nil, nil, domain.NewWizardError(
			fmt.Sprintf("catch error on parse flight_date: %v", queryParameters.Get("flight_date")),
			"bad_date",
		)
	}

	if returnDate != nil && departureDate == nil {
		return nil, nil, nil, domain.NewWizardError("departure date is required for a return date", domain.NoDate)
	}
	return departureDate, returnDate, flightDate, nil
}

var langMapping = map[string]models.Lang{
	"ru": consts.LangRU,
	"be": consts.LangBE,
	"kk": consts.LangKK,
	"tr": consts.LangTR,
	"tt": consts.LangTT,
	"uk": consts.LangUK,
	"en": consts.LangEN,
}

func getLangModel(lang string) models.Lang {
	if model, found := langMapping[lang]; found {
		return model
	}
	return models.Lang(lang)
}

func getOptionalInt(source string) *int {
	if value, err := strconv.Atoi(source); err == nil {
		return &value
	}
	return nil
}

func getOptionalString(source string) *string {
	if source != "" {
		return &source
	}
	return nil
}

func tryParseDate(rawDate string) (*time.Time, error) {
	if rawDate == "" {
		return nil, nil
	}

	date, err := time.Parse("2006-01-02", rawDate)
	if err != nil {
		return nil, fmt.Errorf("failed to parse date: %s", rawDate)
	}
	return &date, nil
}

func getNationalVersionByTld(tld string, flags flags.Flags) string {
	if tld == "by" && flags.ChangeByToRu() {
		return "ru"
	}

	nationalVersionByTldMapper := map[string]string{
		"ru":     "ru",
		"ua":     "ua",
		"kz":     "kz",
		"com.tr": "tr",
		"by":     "by",
		"com":    "com",
	}

	if nationalVersion, found := nationalVersionByTldMapper[tld]; found {
		return nationalVersion
	}
	return "ru"
}
