package finder

import (
	"sort"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type Finder struct {
	index           repositories.NameSearchIndex
	settlementsRepo repositories.Settlement
	stationRepo     repositories.Station
}

func NewFinder(
	index repositories.NameSearchIndex,
	settlementsRepo repositories.Settlement,
	stationRepo repositories.Station,
) *Finder {
	return &Finder{
		index:           index,
		settlementsRepo: settlementsRepo,
		stationRepo:     stationRepo,
	}
}

type settlements []*models.Settlement

func (ss *settlements) ExtendAsInterfaces(objects []interface{}) {
	for _, object := range objects {
		*ss = append(*ss, object.(*models.Settlement))
	}
}

func (ss settlements) Len() int           { return len(ss) }
func (ss settlements) Swap(i, j int)      { ss[i], ss[j] = ss[j], ss[i] }
func (ss settlements) Less(i, j int) bool { return ss[i].MajorityID < ss[j].MajorityID }

func (finder *Finder) FindSettlement(title string) *models.Settlement {
	settlements := make(settlements, 0)
	settlements.ExtendAsInterfaces(finder.index.FindExact(models.SettlementModel, title))
	settlements.ExtendAsInterfaces(finder.index.FindWords(models.SettlementModel, title))

	if byIata, found := finder.settlementsRepo.GetByIataCode(title); found && !byIata.Hidden {
		settlements = append(settlements, byIata)
	}

	if len(settlements) == 0 {
		return nil
	}
	sort.Stable(settlements)
	for _, settlement := range settlements {
		if !finder.settlementsRepo.IsAviaSettlement(settlement.ID) {
			continue
		}
		return settlement
	}
	return nil
}

type stations []*models.Station

func (ss *stations) ExtendAsInterfaces(objects []interface{}) {
	for _, object := range objects {
		*ss = append(*ss, object.(*models.Station))
	}
}

func (ss stations) Len() int           { return len(ss) }
func (ss stations) Swap(i, j int)      { ss[i], ss[j] = ss[j], ss[i] }
func (ss stations) Less(i, j int) bool { return ss[i].MajorityID < ss[j].MajorityID }

func (finder *Finder) FindStation(title string) *models.Station {
	stations := make(stations, 0)
	stations.ExtendAsInterfaces(finder.index.FindExact(models.StationModel, title))
	stations.ExtendAsInterfaces(finder.index.FindWords(models.StationModel, title))

	stations = append(stations, finder.stationRepo.GetByCode(title)...)

	if len(stations) == 0 {
		return nil
	}

	sort.Stable(stations)
	for _, station := range stations {
		if station.Hidden || !station.WithSettlement() || !station.IsAirport() {
			continue
		}
		return station
	}
	return nil
}

func (finder *Finder) FindCountry(title string, exact bool) *models.Country {
	objects := finder.index.FindExact(models.CountryModel, title)
	if len(objects) > 0 {
		return objects[0].(*models.Country)
	}

	if !exact {
		objects = finder.index.FindWords(models.CountryModel, title)
	}
	if len(objects) > 0 {
		return objects[0].(*models.Country)
	}

	return nil
}

func (finder *Finder) FindRegion(title string) *models.Region {
	objects := finder.index.FindExact(models.RegionModel, title)
	if len(objects) > 0 {
		return objects[0].(*models.Region)
	}

	objects = finder.index.FindWords(models.RegionModel, title)
	if len(objects) > 0 {
		return objects[0].(*models.Region)
	}

	return nil
}
