package point

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point/providers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type (
	IParser interface {
		ParseFlightPoints(
			queryParameters *parameters.QueryParameters,
			flightNumber string,
			ctx context.Context,
		) (pointFrom models.Point, pointTo models.Point, err error)
		ParsePointToPoint(
			ctx context.Context,
			queryParameters *parameters.QueryParameters,
		) (fromPoint models.Point, toPoint models.Point, fromSettlement *models.Settlement, toSettlement *models.Settlement, err error)
		PointToSettlement(point models.Point, nationalVersion string, useNearest bool) (*models.Settlement, error)
		PointsIntersect(from models.Point, to models.Point) bool
	}

	Parser struct {
		pointByPointKeyProvider       *providers.PointByPointKeyProvider
		pointByGeoIDProvider          *providers.PointByGeoIDProvider
		pointByTitleProvider          *providers.PointByTitleProvider
		nearestAviaSettlementProvider *providers.NearestAviaSettlementProvider
		settlementsRepository         repositories.Settlement
		stationToSettlementRepository repositories.StationToSettlement
		popularSettlementsRepository  repositories.PopularSettlements
		aviaDirections                repositories.Direction
	}
)

func NewParser(
	pointByKeyProvider *providers.PointByPointKeyProvider,
	providerByGeoID *providers.PointByGeoIDProvider,
	providerByTitle *providers.PointByTitleProvider,
	nearestAviaSettlementProvider *providers.NearestAviaSettlementProvider,
	settlementsRepository repositories.Settlement,
	stationToSettlementRepository repositories.StationToSettlement,
	popularSettlementsRepository repositories.PopularSettlements,
	aviaDirections repositories.Direction,
) *Parser {
	return &Parser{
		pointByPointKeyProvider:       pointByKeyProvider,
		pointByGeoIDProvider:          providerByGeoID,
		pointByTitleProvider:          providerByTitle,
		nearestAviaSettlementProvider: nearestAviaSettlementProvider,
		settlementsRepository:         settlementsRepository,
		stationToSettlementRepository: stationToSettlementRepository,
		popularSettlementsRepository:  popularSettlementsRepository,
		aviaDirections:                aviaDirections,
	}
}

func (parser *Parser) ParsePointToPoint(
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
) (fromPoint models.Point, toPoint models.Point, fromSettlement *models.Settlement, toSettlement *models.Settlement, err error) {
	ppPointsParsingSpan, _ := opentracing.StartSpanFromContext(ctx, "Parsing points for point-to-point")
	defer ppPointsParsingSpan.Finish()

	fromPoint, err = parser.parsePointFromPPParams(
		ctx,
		"from_point",
		queryParameters.FromID,
		queryParameters.FromText,
		queryParameters.GeoID,
		queryParameters.FromGeoID,
		queryParameters,
	)
	if err != nil {
		return fromPoint, toPoint, fromSettlement, toSettlement, err
	}
	toPoint, err = parser.parsePointFromPPParams(
		ctx,
		"to_point",
		queryParameters.ToID,
		queryParameters.ToText,
		-1,
		queryParameters.ToGeoID,
		queryParameters,
	)
	if err != nil {
		return fromPoint, toPoint, fromSettlement, toSettlement, err
	}

	if fromPoint != nil {
		if settlement, err := parser.PointToSettlement(fromPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic()); err != nil {
			err = domain.NewWizardError(fmt.Sprintf("couldn't find settlement for fromPoint: %+v", fromPoint), domain.BadPoint)
			return nil, nil, nil, nil, err
		} else {
			fromSettlement = settlement
		}
	}
	if toPoint != nil {
		if settlement, err := parser.PointToSettlement(toPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic()); err != nil {
			err = domain.NewWizardError(fmt.Sprintf("couldn't find settlement for toPoint: %+v", fromPoint), domain.BadPoint)
			return fromPoint, toPoint, fromSettlement, toSettlement, err
		} else {
			toSettlement = settlement
		}
	} else if toPointByGeoID := parser.getByGeoID(ctx, "to_point", queryParameters.GeoID, queryParameters); toPointByGeoID != nil {
		toSettlementByGeoID, err := parser.PointToSettlement(toPointByGeoID, queryParameters.NationalVersion, !queryParameters.IsDynamic())
		if err != nil {
			err = domain.NewWizardError(fmt.Sprintf("couldn't find settlement for toPointByGeoID: %+v", toPointByGeoID), domain.BadPoint)
			return fromPoint, toPoint, fromSettlement, toSettlement, err
		}
		fromSettlementStations := parser.settlementsRepository.GetStationIDs(fromSettlement.ID)
		toSettlementByGeoIDStations := parser.settlementsRepository.GetStationIDs(toSettlementByGeoID.ID)
		if fromSettlement.ID == toSettlementByGeoID.ID || len(fromSettlementStations.Intersect(toSettlementByGeoIDStations)) > 0 ||
			!parser.aviaDirections.IsPossibleVariant(fromSettlement.ID, toSettlementByGeoID.ID, queryParameters.NationalVersion) {
			toPointByGeoID = nil
			toSettlementByGeoID = nil
		}
		if toPointByGeoID != nil {
			props.SetSearchProp(ctx, "to_point_from_geo_id", "1")
		}
		if queryParameters.Flags.GeoIDAsToID() {
			toSettlement = toSettlementByGeoID
			toPoint = toPointByGeoID
		}
	}
	return fromPoint, toPoint, fromSettlement, toSettlement, err
}

func (parser *Parser) ParseFlightPoints(
	queryParameters *parameters.QueryParameters,
	flightNumber string,
	ctx context.Context,
) (pointFrom models.Point, pointTo models.Point, err error) {
	flightPointsParsingSpan, _ := opentracing.StartSpanFromContext(ctx, "Parsing points for flight")
	defer flightPointsParsingSpan.Finish()
	fromPoint := parser.parsePointFromTitleAndGeoID(queryParameters.FromText, queryParameters.FromGeoID)
	toPoint := parser.parsePointFromTitleAndGeoID(queryParameters.ToText, queryParameters.ToGeoID)

	if helpers.IsDigit(flightNumber) {
		if !helpers.IsNil(fromPoint) && fromPoint.GetPointType() != models.PointTypeStation && fromPoint.GetPointType() != models.PointTypeSettlement {
			if fromSettlement, err := parser.PointToSettlement(fromPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic()); err != nil {
				return nil, nil, fmt.Errorf("couldn't find settlement for fromPoint %+v: %+v", fromPoint, err)
			} else {
				fromPoint = fromSettlement
			}
		}
		if !helpers.IsNil(toPoint) && toPoint.GetPointType() != models.PointTypeStation && toPoint.GetPointType() != models.PointTypeSettlement {
			if toSettlement, err := parser.PointToSettlement(toPoint, queryParameters.NationalVersion, !queryParameters.IsDynamic()); err != nil {
				return nil, nil, fmt.Errorf("couldn't find settlement for toPoint %+v: %+v", toPoint, err)
			} else {
				toPoint = toSettlement
			}
		}
	}
	if parser.PointsIntersect(fromPoint, toPoint) {
		toPoint = nil
	}
	return fromPoint, toPoint, nil
}

func (parser *Parser) parsePointFromTitleAndGeoID(title string, geoID *int) models.Point {
	if point := parser.pointByTitleProvider.Get(title); !helpers.IsNil(point) {
		return point
	}
	if geoID != nil && *geoID > 0 {
		return parser.pointByGeoIDProvider.Get(*geoID)
	}
	return nil
}

func (parser *Parser) PointToSettlement(point models.Point, nationalVersion string, useNearest bool) (*models.Settlement, error) {
	var (
		settlement *models.Settlement
		found      bool
	)
	if helpers.IsNil(point) {
		return nil, domain.NewWizardError("point is nil", domain.BadPoint)
	}
	switch p := point.(type) {
	case *models.Station:
		if !helpers.IsNil(p) {
			settlement, found = parser.settlementsRepository.GetByID(p.SettlementID)
			if !found {
				if settlements := parser.stationToSettlementRepository.GetSettlementsByStationID(p.ID); len(settlements) > 0 {
					settlement = settlements[0]
					found = true
				}
			}
		}
	case *models.Country:
		settlement, found = parser.popularSettlementsRepository.GetPopularByCountry(nationalVersion, p.ID)
		if !found {
			settlement, _ = parser.settlementsRepository.GetCapital(p.ID)
		}
	case *models.Region:
		settlement, found = parser.popularSettlementsRepository.GetPopularByRegion(nationalVersion, p.ID)
		if !found {
			settlement, _ = parser.settlementsRepository.GetRegionCapital(p.ID)
		}
	default:
		settlement = point.(*models.Settlement)
	}

	if helpers.IsNil(settlement) {
		return nil, domain.NewWizardError(fmt.Sprintf("point without a settlement: %+v", point), domain.NoSettlement)
	}

	return parser.substitute(settlement, useNearest), nil
}

func (parser *Parser) substitute(settlement *models.Settlement, useNearest bool) *models.Settlement {
	if parser.settlementsRepository.IsAviaSettlement(settlement.GetID()) || !useNearest {
		return settlement
	}
	if nearest, found := parser.nearestAviaSettlementProvider.GetNearest(settlement); found {
		return nearest
	}
	if center, found := parser.settlementsRepository.GetAviaRegionCenter(settlement.RegionID); found {
		return center
	}
	if capital, found := parser.settlementsRepository.GetAviaCapital(settlement.CountryID); found {
		return capital
	}
	return settlement
}

func (parser *Parser) PointsIntersect(from models.Point, to models.Point) bool {
	if helpers.IsNil(from) || helpers.IsNil(to) {
		return false
	}
	if from.GetPointType() != models.PointTypeStation && from.GetPointType() != models.PointTypeSettlement ||
		to.GetPointType() != models.PointTypeStation && to.GetPointType() != models.PointTypeSettlement {
		return false
	}
	fromStations := parser.getStationIDsForPoint(from)
	toStations := parser.getStationIDsForPoint(to)
	return len(fromStations.Intersect(toStations)) > 0
}

func (parser *Parser) getStationIDsForPoint(point models.Point) containers.SetOfInt {
	stationIDs := make(containers.SetOfInt)
	if point.GetPointType() == models.PointTypeStation {
		stationIDs.Add(point.GetID())
		return stationIDs
	}

	return parser.settlementsRepository.GetStationIDs(point.GetID())
}

func (parser *Parser) parsePointFromPPParams(
	ctx context.Context,
	direction string,
	pointKey *string,
	pointTitle string,
	userGeoID int,
	garGeoID *int,
	queryParameters *parameters.QueryParameters,
) (models.Point, error) {
	pointByTitle := parser.pointByTitleProvider.Get(pointTitle)
	pointByUserGeoID := parser.getByGeoID(ctx, direction, userGeoID, queryParameters)
	var pointByGarGeoID models.Point
	canParseGarGeoIDAsRegion := false
	if garGeoID != nil {
		pointByGarGeoID = parser.pointByGeoIDProvider.Get(*garGeoID)
		canParseGarGeoIDAsRegion = parser.pointByGeoIDProvider.GetRegion(*garGeoID) != nil
	}
	regionByTitle := parser.pointByTitleProvider.FindRegion(pointTitle)

	props.SetSearchProp(ctx, fmt.Sprintf("%s_can_parse_gar_geo_id_as_region", direction), canParseGarGeoIDAsRegion)
	props.SetSearchProp(ctx, fmt.Sprintf("%s_can_parse_by_title", direction), !helpers.IsNil(pointByTitle))
	props.SetSearchProp(ctx, fmt.Sprintf("%s_can_parse_by_geo_id", direction), !helpers.IsNil(pointByGarGeoID))
	props.SetSearchProp(ctx, fmt.Sprintf("%s_can_parse_title_as_region", direction), !helpers.IsNil(regionByTitle))

	if pointByKey := parser.pointByPointKeyProvider.Get(pointKey); pointByKey != nil {
		return pointByKey, nil
	}
	if pointByTitle != nil {
		return pointByTitle, nil
	}
	if pointByGarGeoID != nil {
		return pointByGarGeoID, nil
	}
	if pointByUserGeoID != nil {
		return pointByUserGeoID, nil
	}
	if pointTitle != "" {
		return nil, domain.NewWizardError(
			fmt.Sprintf("failed to choose point: title = %s, userGeoID = %d, garGeoID = %v", pointTitle, userGeoID, garGeoID),
			"unknown_points",
		)
	}
	return nil, nil
}

func (parser *Parser) getByGeoID(
	ctx context.Context,
	direction string,
	geoID int,
	queryParameters *parameters.QueryParameters,
) models.Point {
	point := parser.pointByGeoIDProvider.Get(geoID)
	if helpers.IsNil(point) {
		pointFromGeobase := parser.pointByGeoIDProvider.GetWithGeoBase(geoID, queryParameters.NationalVersion)
		if queryParameters.Flags.GeoIDLookup() {
			props.SetSearchProp(ctx, fmt.Sprintf("%s_parsed_with_geobase", direction), !helpers.IsNil(pointFromGeobase))
			point = pointFromGeobase
		} else {
			props.SetSearchProp(ctx, fmt.Sprintf("%s_could_parse_with_geobase", direction), !helpers.IsNil(pointFromGeobase))
		}
	}
	return point
}
