package providers

import (
	"github.com/golang/geo/s1"
	"github.com/golang/geo/s2"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type NearestAviaSettlementProvider struct {
	settlementRepository repositories.Settlement
	settlementIndex      *s2.ShapeIndex
	edgeIDToSettlement   map[int]*models.Settlement
	radiusKM             float32
}

func NewNearestAviaSettlementProvider(settlementRepository repositories.Settlement, radiusKM float32) *NearestAviaSettlementProvider {
	index := s2.NewShapeIndex()
	points := s2.PointVector{}
	edgeIDToSettlement := make(map[int]*models.Settlement)
	for _, settlement := range settlementRepository.GetAll() {
		if settlementRepository.IsAviaSettlement(settlement.ID) && (settlement.Longitude != 0 || settlement.Latitude != 0) {
			point := s2.PointFromLatLng(s2.LatLngFromDegrees(settlement.Latitude, settlement.Longitude))
			points = append(points, point)
			edgeIDToSettlement[len(points)-1] = settlement
		}
	}
	index.Add(&points)
	return &NearestAviaSettlementProvider{
		settlementRepository: settlementRepository,
		settlementIndex:      index,
		edgeIDToSettlement:   edgeIDToSettlement,
		radiusKM:             radiusKM,
	}
}

func (nearestAviaSettlementProvider *NearestAviaSettlementProvider) GetNearest(settlement *models.Settlement) (*models.Settlement, bool) {
	q := s2.NewClosestEdgeQuery(
		nearestAviaSettlementProvider.settlementIndex,
		s2.NewClosestEdgeQueryOptions().
			MaxResults(1).
			DistanceLimit(s1.ChordAngleFromAngle(kmToAngle(nearestAviaSettlementProvider.radiusKM))),
	)
	point := s2.PointFromLatLng(s2.LatLngFromDegrees(settlement.Latitude, settlement.Longitude))
	target := s2.NewMinDistanceToPointTarget(point)
	result := q.FindEdges(target)
	if len(result) > 0 {
		return nearestAviaSettlementProvider.edgeIDToSettlement[int(result[0].EdgeID())], true
	}
	return nil, false
}

func (nearestAviaSettlementProvider *NearestAviaSettlementProvider) GetDistanceInKMBetween(from, to *models.Settlement) *float64 {
	if (from.Latitude == 0 && from.Longitude == 0) || (to.Latitude == 0 && to.Longitude == 0) {
		return nil
	}
	pointFrom := s2.PointFromLatLng(s2.LatLngFromDegrees(from.Latitude, from.Longitude))
	pointTo := s2.PointFromLatLng(s2.LatLngFromDegrees(to.Latitude, to.Longitude))
	km := angleToKM(pointFrom.Distance(pointTo))
	return ptr.Float64(km)
}

// The Earth's mean radius in kilometers (according to NASA).
const earthRadiusKm = 6371.01

func kmToAngle(km float32) s1.Angle {
	return s1.Angle(km / earthRadiusKm)
}

func angleToKM(angle s1.Angle) float64 {
	// The Earth's mean radius in kilometers (according to NASA).
	return earthRadiusKm * float64(angle)
}
