package providers

import (
	"a.yandex-team.ru/library/go/yandex/geobase"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type PointByGeoIDProvider struct {
	countryRepo    repositories.Country
	settlementRepo repositories.Settlement
	regionRepo     repositories.Region
	geobaseClient  geobase.Geobase
}

func NewPointProviderByGeoID(
	countryRepo repositories.Country,
	settlementRepo repositories.Settlement,
	regionRepo repositories.Region,
	geobaseClient geobase.Geobase,
) *PointByGeoIDProvider {
	return &PointByGeoIDProvider{
		countryRepo:    countryRepo,
		settlementRepo: settlementRepo,
		regionRepo:     regionRepo,
		geobaseClient:  geobaseClient,
	}
}

func (provider *PointByGeoIDProvider) Get(geoID int) models.Point {
	if geoID < 0 {
		return nil
	}
	if point, found := provider.settlementRepo.GetByGeoID(geoID); found {
		return point
	}
	if point, found := provider.regionRepo.GetByGeoID(geoID); found {
		return point
	}
	if point, found := provider.countryRepo.GetByID(geoID); found {
		return point
	}
	return nil
}

func (provider *PointByGeoIDProvider) GetWithGeoBase(geoID int, nationalVersion string) models.Point {
	if point := provider.Get(geoID); !helpers.IsNil(point) {
		return point
	}
	crimeaStatus := geobase.CrimeaInRU
	if nationalVersion != "ru" {
		crimeaStatus = geobase.CrimeaInUA
	}
	if !helpers.IsNil(provider.geobaseClient) {
		parents, err := provider.geobaseClient.GetParentsIDs(geobase.ID(geoID), crimeaStatus)
		if err != nil {
			return nil
		}
		for _, parentID := range parents {
			if point := provider.Get(int(parentID)); !helpers.IsNil(point) {
				return point
			}
		}
	}
	return nil
}

func (provider *PointByGeoIDProvider) GetRegion(geoID int) models.Point {
	if point, found := provider.regionRepo.GetByGeoID(geoID); found {
		return point
	}
	return nil
}
