package search

import (
	"context"
	"errors"
	"strconv"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
	personalsearch "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	contextlib "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/context"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	resultslib "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories/ydb"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/personalization"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/tdapi"
	"a.yandex-team.ru/travel/proto/avia/wizard"
)

type ResultProvider struct {
	appLogger               log.Logger
	wizardResultsRepository repositories.ResultsRepository
	ticketDaemonAPIClient   tdapi.Client
	personalizationClient   personalization.Client
	resultsMerger           *resultslib.MergerByConversion
	heatCache               bool
	writeStats              bool
	usePartnerCache         bool
}

func NewResultProvider(
	appLogger log.Logger,
	wizardResultsRepository repositories.ResultsRepository,
	ticketDaemonAPIClient tdapi.Client,
	personalizationClient personalization.Client,
	resultsMerger *resultslib.MergerByConversion,
	heatCache bool,
	writeStats bool,
	usePartnerCache bool,
) *ResultProvider {
	return &ResultProvider{
		appLogger:               appLogger,
		wizardResultsRepository: wizardResultsRepository,
		ticketDaemonAPIClient:   ticketDaemonAPIClient,
		personalizationClient:   personalizationClient,
		resultsMerger:           resultsMerger,
		heatCache:               heatCache,
		writeStats:              writeStats,
		usePartnerCache:         usePartnerCache,
	}
}

func (resultProvider *ResultProvider) GetWizardResult(
	queryParameters *parameters.QueryParameters,
	departureDate *time.Time,
	fromSettlement, toSettlement *models.Settlement,
	now time.Time,
	ctx context.Context,
) (*ydb.WizardSearchResult, error) {
	ctx = contextlib.WithPartnerCache(ctx, queryParameters.Flags.EnablePartnerCache())

	var err error
	var wizardResults []*ydb.WizardSearchResult

	cacheTypes := make([]string, 0)
	defer func() {
		if len(cacheTypes) > 0 {
			if resultProvider.writeStats {
				if slices.ContainsString(cacheTypes, consts.WizardResultsExperimental) {
					metrics.GlobalWizardMetrics().GetExperimentalCacheMissCounter(queryParameters.IsDynamic()).Inc()
					metrics.GlobalWizardMetrics().GetExperimentalCacheMissByNationalVersionCounter(
						queryParameters.NationalVersion,
						queryParameters.IsDynamic(),
					).Inc()
				}
				if slices.ContainsString(cacheTypes, consts.WizardResults) {
					metrics.GlobalWizardMetrics().GetCacheMissCounter(queryParameters.IsDynamic()).Inc()
					metrics.GlobalWizardMetrics().GetCacheMissByNationalVersionCounter(
						queryParameters.NationalVersion,
						queryParameters.IsDynamic(),
					).Inc()
				}
			}
			if resultProvider.heatCache && !helpers.IsUkraine(fromSettlement, toSettlement) {
				resultProvider.heatCacheMiss(
					fromSettlement,
					toSettlement,
					departureDate,
					queryParameters,
					ctx,
					cacheTypes,
				)
			}
		}
	}()
	if queryParameters.IsDynamic() {
		if !queryParameters.Flags.SearchDepthExperiment() {
			wizardResults, err = resultProvider.getDynamic(
				queryParameters,
				fromSettlement,
				toSettlement,
				departureDate,
				now,
				ctx,
			)
			if err == nil && len(wizardResults) == 0 {
				cacheTypes = append(cacheTypes, consts.WizardResults)
			}
		} else {
			wizardResults, err = resultProvider.getExperimentalDynamic(
				queryParameters,
				fromSettlement,
				toSettlement,
				departureDate,
				now,
				ctx,
			)
			if err == nil && len(wizardResults) != 0 {
				cacheTypes = append(cacheTypes, consts.WizardResultsExperimental)
			}
		}
		wizardResult := resultProvider.resultsMerger.MergeByConversion(wizardResults)
		if wizardResult != nil && helpers.IsUkraine(fromSettlement, toSettlement) {
			if wizardResult.SearchResult.Value != nil {
				wizardResult.SearchResult.Value.Fares = []*wizard.Fare{}
			}
			wizardResult.FilterState.Value = results.FilterState{}
		}
		return wizardResult, err
	}
	wg := sync.WaitGroup{}
	wg.Add(2)
	var expWizardResult []*ydb.WizardSearchResult
	var expErr error
	var controlWizardResult []*ydb.WizardSearchResult
	var controlErr error
	go func() {
		defer wg.Done()
		if !queryParameters.Flags.EnableStaticExperimentalQueries() {
			return
		}
		expWizardResult, expErr = resultProvider.getExperimentalStatic(
			queryParameters,
			fromSettlement,
			toSettlement,
			departureDate,
			now,
			ctx,
		)
		if expErr == nil && len(expWizardResult) == 0 {
			cacheTypes = append(cacheTypes, consts.WizardResultsExperimental)
		}
	}()
	go func() {
		defer wg.Done()
		if queryParameters.Flags.PersonalizedDefaultDayOrMinPriceDate() && queryParameters.DepartureDate == nil {
			controlWizardResult, controlErr = resultProvider.getPersonalizedStatic(
				queryParameters,
				fromSettlement,
				toSettlement,
				departureDate,
				now,
				ctx,
			)
		} else {
			controlWizardResult, controlErr = resultProvider.getStatic(
				queryParameters,
				fromSettlement,
				toSettlement,
				departureDate,
				now,
				ctx,
			)
		}
		if controlErr == nil && len(controlWizardResult) == 0 {
			cacheTypes = append(cacheTypes, consts.WizardResults)
		}
	}()
	wg.Wait()
	if queryParameters.Flags.SearchDepthExperiment() {
		return resultProvider.mergeResults(expWizardResult, expErr)
	}
	return resultProvider.mergeResults(controlWizardResult, controlErr)
}

func ConvertDateToDays(t *time.Time) uint32 {
	if t == nil {
		return 0
	}
	return uint32(t.Unix() / consts.SecondsInDay)
}

func getUserNowInDays(t *time.Time, userTime time.Time) uint32 {
	if t == nil {
		return 0
	}
	_, zoneSecondsOffset := userTime.Zone()
	return uint32((t.Unix() + int64(zoneSecondsOffset)) / consts.SecondsInDay)
}

func (resultProvider *ResultProvider) heatCacheMiss(
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	queryParameters *parameters.QueryParameters,
	ctx context.Context,
	cacheTypes []string,
) {
	if departureDate == nil {
		departureDate = helpers.PredictDate(queryParameters.UserTime)
	}
	go resultProvider.ticketDaemonAPIClient.InitSearch(
		&tdapi.InitSearchRequest{
			PointKeyFrom:    fromSettlement.GetPointKey(),
			PointKeyTo:      toSettlement.GetPointKey(),
			DateForward:     *departureDate,
			QueryParameters: queryParameters,
			Context:         ctx,
			CacheTypes:      cacheTypes,
		},
	)
}

func (resultProvider *ResultProvider) getStatic(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	provider := newResultByPartnerFetcher(
		resultProvider.appLogger,
		"Static",
		resultProvider.getStaticByAllPartners,
		resultProvider.getStaticByPartner,
		resultProvider.usePartnerCache,
	)
	return provider.getResults(queryParameters, fromSettlement, toSettlement, departureDate, now, ctx)
}

func (resultProvider *ResultProvider) getStaticByAllPartners(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	return resultProvider.wizardResultsRepository.GetStatic(
		&repositories.StaticQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			NowInDays:       getUserNowInDays(&now, queryParameters.UserTime),
			Now:             uint64(now.UTC().Unix()),
			DepartureDate:   ConvertDateToDays(departureDate),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getStaticByPartner(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	defer func() {
		if e := recover(); e != nil {
			wizardResult = nil
			err = recoverError(e)
		}
	}()
	return resultProvider.wizardResultsRepository.GetStaticByPartner(
		&repositories.StaticQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			NowInDays:       getUserNowInDays(&now, queryParameters.UserTime),
			Now:             uint64(now.UTC().Unix()),
			DepartureDate:   ConvertDateToDays(departureDate),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func recoverError(e interface{}) error {
	switch x := e.(type) {
	case string:
		return errors.New(x)
	case error:
		return x
	default:
		return errors.New("unknown panic")
	}
}

func (resultProvider *ResultProvider) getDynamic(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	if departureDate == nil {
		departureDate = helpers.PredictDate(queryParameters.UserTime)
	}

	provider := newResultByPartnerFetcher(
		resultProvider.appLogger,
		"Dynamic",
		resultProvider.getDynamicByAllPartners,
		resultProvider.getDynamicByPartners,
		resultProvider.usePartnerCache,
	)
	return provider.getResults(queryParameters, fromSettlement, toSettlement, departureDate, now, ctx)
}

func (resultProvider *ResultProvider) getDynamicByAllPartners(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	return resultProvider.wizardResultsRepository.GetDynamic(
		&repositories.DynamicQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			DepartureDate:   ConvertDateToDays(departureDate),
			ReturnDate:      ConvertDateToDays(queryParameters.ReturnDate),
			Now:             uint64(now.UTC().Unix()),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getDynamicByPartners(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResults []*ydb.WizardSearchResult, err error) {
	defer func() {
		if e := recover(); e != nil {
			wizardResults = nil
			err = recoverError(e)
		}
	}()
	return resultProvider.wizardResultsRepository.GetDynamicByPartners(
		&repositories.DynamicQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			DepartureDate:   ConvertDateToDays(departureDate),
			ReturnDate:      ConvertDateToDays(queryParameters.ReturnDate),
			Now:             uint64(now.UTC().Unix()),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getExperimentalDynamic(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	if departureDate == nil {
		departureDate = helpers.PredictDate(queryParameters.UserTime)
	}
	return resultProvider.wizardResultsRepository.GetExperimentalDynamic(
		&repositories.DynamicQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			DepartureDate:   ConvertDateToDays(departureDate),
			ReturnDate:      ConvertDateToDays(queryParameters.ReturnDate),
			Now:             uint64(now.UTC().Unix()),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getExperimentalStatic(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	return resultProvider.wizardResultsRepository.GetExperimentalStatic(
		&repositories.StaticQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			NowInDays:       getUserNowInDays(&now, queryParameters.UserTime),
			DepartureDate:   ConvertDateToDays(departureDate),
			Now:             uint64(now.UTC().Unix()),
			JobID:           queryParameters.JobID,
			UseTTLParallel:  queryParameters.Flags.SearchDepthExperiment() && queryParameters.Flags.UseTTLParallelRequests(),
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getPersonalizedStatic(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	provider := newResultByPartnerFetcher(
		resultProvider.appLogger,
		"PersonalizedStatic",
		resultProvider.getPersonalizedStaticByAllPartners,
		resultProvider.getPersonalizedStaticByPartner,
		resultProvider.usePartnerCache,
	)
	return provider.getResults(queryParameters, fromSettlement, toSettlement, departureDate, now, ctx)
}

func (resultProvider *ResultProvider) getPersonalizedStaticByAllPartners(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	return resultProvider.wizardResultsRepository.GetPersonalizedStatic(
		&repositories.StaticQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			NowInDays:       getUserNowInDays(&now, queryParameters.UserTime),
			Now:             uint64(now.UTC().Unix()),
			DepartureDate:   ConvertDateToDays(departureDate),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) getPersonalizedStaticByPartner(
	queryParameters *parameters.QueryParameters,
	fromSettlement, toSettlement *models.Settlement,
	departureDate *time.Time,
	now time.Time,
	ctx context.Context,
) (wizardResult []*ydb.WizardSearchResult, err error) {
	defer func() {
		if e := recover(); e != nil {
			wizardResult = nil
			err = recoverError(e)
		}
	}()
	return resultProvider.wizardResultsRepository.GetPersonalizedStaticByPartner(
		&repositories.StaticQuery{
			PointFrom:       fromSettlement.GetPointKey(),
			PointTo:         toSettlement.GetPointKey(),
			Klass:           1,
			Passengers:      queryParameters.Passengers().ToUInt32(),
			NationalVersion: queryParameters.NationalVersion,
			NowInDays:       getUserNowInDays(&now, queryParameters.UserTime),
			Now:             uint64(now.UTC().Unix()),
			DepartureDate:   ConvertDateToDays(departureDate),
			JobID:           queryParameters.JobID,
		}, ctx,
	)
}

func (resultProvider *ResultProvider) GetPersonalSearchResult( //TODO: унести кудато. К ResultProvider метод не имеет никакого отношения
	ctx context.Context,
	queryParameters *parameters.QueryParameters,
) (personalSearchResult *personalsearch.TGetPersonalSearchResponseV2, err error) {
	geoID := strconv.Itoa(queryParameters.GeoID)

	return resultProvider.personalizationClient.SendGetPersonalSearch(
		ctx,
		queryParameters.YandexUID,
		geoID,
		queryParameters.JobID,
	)
}

func (resultProvider ResultProvider) mergeResults(wizardResults []*ydb.WizardSearchResult, err error) (*ydb.WizardSearchResult, error) {
	if err != nil {
		return nil, err
	}
	return resultProvider.resultsMerger.MergeByConversion(wizardResults), nil
}
