package handlers

import (
	"context"
	"net/http"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/library/go/utils"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	domainHandlers "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers"
	domainResponses "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

type VariantsInfoHandler struct {
	appLogger             log.Logger
	responseJSONLogger    log.Logger
	handler               *domainHandlers.VariantsInfoHandler
	queryParametersParser parameters.IQueryParametersParser
	settings              *settings.Settings
}

func NewVariantsInfoHandler(
	appLogger log.Logger,
	responseJSONLogger log.Logger,
	settings *settings.Settings,
	handler *domainHandlers.VariantsInfoHandler,
	queryParametersParser parameters.IQueryParametersParser,
) *VariantsInfoHandler {
	return &VariantsInfoHandler{
		appLogger:             appLogger,
		responseJSONLogger:    responseJSONLogger,
		settings:              settings,
		handler:               handler,
		queryParametersParser: queryParametersParser,
	}
}

func (handler *VariantsInfoHandler) GetRoute() string {
	return "/api/cached_variants/"
}

var badRequestCodes = map[string]interface{}{
	domain.BadArguments:       nil,
	domain.BadPoint:           nil,
	domain.BadNationalVersion: nil,
	domain.BadBaggageCode:     nil,
	domain.BadTld:             nil,
	domain.BadLang:            nil,
	domain.BadSettlement:      nil,
	domain.BadFlightNumber:    nil,
	domain.BadFlight:          nil,
	domain.BadFlightPoints:    nil,
	domain.SameAirports:       nil,
	domain.SamePoint:          nil,
}

func (handler *VariantsInfoHandler) ServeHTTP(writer http.ResponseWriter, request *http.Request) {
	ctx := request.Context()
	handlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "variants info handler")
	defer handlerSpan.Finish()

	startTime := time.Now()
	queryParameters, err := handler.queryParametersParser.Parse(request)
	if err != nil {
		resp := domainResponses.NewErrorResponseFromString(err.Error(), nil, nil)
		utils.HTTP.JSON(writer, http.StatusBadRequest, resp)
		return
	}
	ctx, cancelFunc := context.WithTimeout(
		ctx,
		handler.settings.MaxVariantsInfoRequestDuration-time.Since(startTime),
	)
	defer cancelFunc()

	landingParams := map[string]string{
		"lang": queryParameters.Lang.String(),
	}

	resp, err := handler.handler.Handle(ctx, queryParameters, landingParams)
	if err != nil {
		switch e := err.(type) {
		case *domain.WizardError:
			resp := domainResponses.NewErrorResponse(err.Error(), e.Code)
			if _, ok := badRequestCodes[e.Code]; ok {
				utils.HTTP.JSON(writer, http.StatusBadRequest, resp)
			} else {
				utils.HTTP.JSON(writer, http.StatusInternalServerError, resp)
			}
		default:
			resp := domainResponses.NewErrorResponseFromError(err, nil, nil)
			utils.HTTP.JSON(writer, http.StatusInternalServerError, resp)
		}
		return
	}
	utils.HTTP.JSON(writer, http.StatusOK, resp)
}
