package handlers

import (
	"encoding/json"
	"net/http"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/travel/avia/library/go/utils"
	domainHandlers "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	aviaMetrics "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/middlewares"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type WizardHandler struct {
	appLogger          log.Logger
	responseJSONLogger log.Logger
	wizardHandler      *domainHandlers.WizardHandler
	settings           *settings.Settings
}

func (handler *WizardHandler) ServeHTTP(writer http.ResponseWriter, request *http.Request) {
	ctx := request.Context()
	wizardHandlerSpan, ctx := opentracing.StartSpanFromContext(ctx, "wizard handler")
	defer wizardHandlerSpan.Finish()

	handlerResponse, errorResponse := handler.wizardHandler.Handle(request.WithContext(ctx))
	serializingResponseSpan, _ := opentracing.StartSpanFromContext(ctx, "Serializing wizard response to JSON")
	defer serializingResponseSpan.Finish()
	if helpers.IsNil(errorResponse) {
		utils.HTTP.JSON(writer, http.StatusOK, handlerResponse)
	} else {
		code := http.StatusOK
		if errorResponse.Code == "no_flights" {
			code = http.StatusNotFound
		}
		utils.HTTP.JSON(writer, code, errorResponse)
	}
	if handler.settings.DebugLogResponseJSON {
		go func() {
			var responseJSON []byte
			if helpers.IsNil(errorResponse) {
				responseJSON, _ = json.Marshal(handlerResponse)
			} else {
				responseJSON, _ = json.Marshal(errorResponse)
			}
			handler.responseJSONLogger.Info(string(responseJSON))
		}()
	}
}

func (handler *WizardHandler) GetRoute() string {
	return "/api/wizard/avia/"
}

func NewWizardHandler(
	appLogger log.Logger,
	responseJSONLogger log.Logger,
	settings *settings.Settings,
	wizardHandler *domainHandlers.WizardHandler,
) *WizardHandler {
	return &WizardHandler{
		appLogger:          appLogger,
		responseJSONLogger: responseJSONLogger,
		settings:           settings,
		wizardHandler:      wizardHandler,
	}
}

func GetWizardHandlerMiddlewares(
	jobIDGenerator helpers.JobIDGenerator,
	appLogger log.Logger,
	tracer opentracing.Tracer,
	httpMetricsRegistry *solomon.Registry,
	maxInflightRequests uint32,
) []func(next http.Handler) http.Handler {
	return []func(next http.Handler) http.Handler{
		middlewares.NewRecoverMiddleware(appLogger).Handle,
		httpmetrics.New(httpMetricsRegistry, httpmetrics.WithPathEndpoint(), httpmetrics.WithSolomonRated()),
		middlewares.NewReqIDMetricsMiddleware().Handle,
		middlewares.NewReqIDFilterMiddleware(jobIDGenerator).Handle,
		middlewares.NewTracingMiddleware(tracer).Handle,
		metrics.NewMetricsMiddleware(aviaMetrics.GlobalWizardMetrics().RequestTimer).Handle,
		middlewares.NewSearchPropsMiddleware(appLogger).Handle,
		middlewares.NewContextPropsMiddleware(appLogger).Handle,
		middlewares.NewTooManyRequestsMiddleware(appLogger, maxInflightRequests).Handle,
	}
}

func GetVariantsInfoHandlerMiddlewares(
	appLogger log.Logger,
	tracer opentracing.Tracer,
	httpMetricsRegistry *solomon.Registry,
	maxInflightRequests uint32,
) []func(next http.Handler) http.Handler {
	return []func(next http.Handler) http.Handler{
		middlewares.NewRecoverMiddleware(appLogger).Handle,
		httpmetrics.New(httpMetricsRegistry, httpmetrics.WithPathEndpoint(), httpmetrics.WithSolomonRated()),
		middlewares.NewTracingMiddleware(tracer).Handle,
		middlewares.NewSearchPropsMiddleware(appLogger).Handle,
		middlewares.NewContextPropsMiddleware(appLogger).Handle,
		middlewares.NewTooManyRequestsMiddleware(appLogger, maxInflightRequests).Handle,
	}
}
