package helpers

import (
	"fmt"
	"time"
)

func TruncateToDate(dt time.Time) time.Time {
	year, month, day := dt.Date()
	return time.Date(year, month, day, 0, 0, 0, 0, dt.Location())
}

func ReplaceLocation(date time.Time, location *time.Location) time.Time {
	return time.Date(
		date.Year(),
		date.Month(),
		date.Day(),
		date.Hour(),
		date.Minute(),
		date.Second(),
		date.Nanosecond(),
		location,
	)
}

func SplitDuration(duration time.Duration) (days, hours, minutes, seconds int) {
	days = int(duration.Hours() / 24)
	hours = int(duration.Hours()) % 24
	minutes = int(duration.Minutes()) % 60
	seconds = int(duration.Seconds()) % 60
	return days, hours, minutes, seconds
}

func RoundDuration(duration time.Duration) time.Duration {
	days, hours, minutes, _ := SplitDuration(duration)
	if days > 0 {
		return time.Duration(days) * 24 * time.Hour
	}
	if hours > 0 {
		return time.Duration(hours) * time.Hour
	}
	return time.Duration(minutes) * time.Minute
}

func PredictDate(now time.Time) *time.Time {
	weekday := now.Weekday()
	var prediction time.Time
	switch weekday {
	case time.Monday, time.Tuesday, time.Wednesday, time.Saturday, time.Sunday:
		prediction = now.Add(24 * time.Hour)
	default:
		daysCount := time.Duration(5 - int(weekday) + 1)
		prediction = now.Add(daysCount * 24 * time.Hour)
	}
	return &prediction
}

// Format: "YYYY-mm-dd HH:MM:SS"
func ParseTimeInLocation(date string, timezone *time.Location) (t time.Time, err error) {
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("bad time format: %s", date)
		}
	}()
	year := (((int(date[0])-'0')*10+int(date[1])-'0')*10+int(date[2])-'0')*10 + int(date[3]) - '0'
	month := time.Month((int(date[5])-'0')*10 + int(date[6]) - '0')
	day := (int(date[8])-'0')*10 + int(date[9]) - '0'
	hour := (int(date[11])-'0')*10 + int(date[12]) - '0'
	minute := (int(date[14])-'0')*10 + int(date[15]) - '0'
	second := (int(date[17])-'0')*10 + int(date[18]) - '0'
	return time.Date(year, month, day, hour, minute, second, 0, timezone), nil
}
