package results

import (
	"math"
	"sort"
	"strings"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories/ydb"
	"a.yandex-team.ru/travel/proto/avia/wizard"
)

type ConversionRepository interface {
	GetAll() repositories.Conversions
}

type MergerByConversion struct {
	conversionRepository ConversionRepository
}

func NewMergerByConversion(conversionRepository ConversionRepository) *MergerByConversion {
	return &MergerByConversion{conversionRepository: conversionRepository}
}

func (m *MergerByConversion) MergeByConversion(values []*ydb.WizardSearchResult) *ydb.WizardSearchResult {
	if len(values) == 0 {
		return nil
	}
	if len(values) == 1 {
		return values[0]
	}

	fares := m.mergeFares(values)
	return m.buildResult(values, fares)
}

type fareGroup struct {
	popularity int32
	fares      []*wizard.Fare
	route      string
}

func (m *MergerByConversion) mergeFares(values []*ydb.WizardSearchResult) []*wizard.Fare {
	fareGroups := getSortedFareGroups(makeFaresByRoute(values))
	partnerConversions := m.copyPartnerConversion()

	result := make([]*wizard.Fare, 0, len(fareGroups))
	for _, group := range fareGroups {
		bestConversionFare := getBestConversionFare(group, partnerConversions)
		cheapestFare := getCheapestFare(group)
		resultFare := cheapestFare
		if bestConversionFare != nil {
			resultFare = bestConversionFare
		}
		delete(partnerConversions, resultFare.GetPartner())
		if resultFare != nil {
			result = append(result, resultFare)
		}
	}
	return result
}

func (m *MergerByConversion) copyPartnerConversion() map[string]float32 {
	partnerConversions := make(map[string]float32)
	for k, v := range m.conversionRepository.GetAll() {
		if len(k) == 0 {
			continue
		}
		partnerConversions[k] = v
	}
	return partnerConversions
}

func getBestConversionFare(group *fareGroup, partnerConversions map[string]float32) *wizard.Fare {
	var bestConversionValue float32
	var bestConversionFare *wizard.Fare
	for _, fare := range group.fares {
		conversionValue, conversionFound := partnerConversions[fare.GetPartner()]
		if conversionFound && bestConversionValue < conversionValue {
			bestConversionValue = conversionValue
			bestConversionFare = fare
		}
	}
	return bestConversionFare
}

func getCheapestFare(group *fareGroup) *wizard.Fare {
	var cheapestPrice float64
	var cheapestFare *wizard.Fare
	for _, fare := range group.fares {
		price := fare.GetTariff().GetValue()
		if price == 0 {
			continue
		}
		if cheapestPrice == 0 || price < cheapestPrice {
			cheapestPrice = price
			cheapestFare = fare
		}
	}
	return cheapestFare
}

func getSortedFareGroups(faresByRoute map[string]*fareGroup) []*fareGroup {
	groups := make([]*fareGroup, 0, len(faresByRoute))
	for _, group := range faresByRoute {
		groups = append(groups, group)
	}
	sort.Slice(groups, func(i, j int) bool {
		return groups[i].popularity > groups[j].popularity
	})
	return groups
}

func makeFaresByRoute(values []*ydb.WizardSearchResult) map[string]*fareGroup {
	faresByRoute := make(map[string]*fareGroup)
	for _, value := range values {
		for _, fare := range value.SearchResult.Value.GetFares() {
			route := makeRouteKey(fare.GetRoute())
			group, found := faresByRoute[route]
			if !found {
				group = &fareGroup{
					popularity: fare.GetPopularity(),
					route:      route,
				}
			}
			group.fares = append(group.fares, fare)
			faresByRoute[route] = group
		}
	}
	return faresByRoute
}

func makeRouteKey(route *wizard.RouteSegments) string {
	if route == nil || len(route.Forward)+len(route.Backward) == 0 {
		return ""
	}
	var buf strings.Builder

	serializer := func(segments []string) {
		if len(segments) == 0 {
			return
		}
		buf.WriteString(segments[0])
		for _, segment := range segments[1:] {
			buf.WriteString("-")
			buf.WriteString(segment)
		}
	}
	serializer(route.Forward)
	buf.WriteString("|")
	serializer(route.Backward)
	return buf.String()
}

func (m *MergerByConversion) buildResult(values []*ydb.WizardSearchResult, fares []*wizard.Fare) *ydb.WizardSearchResult {
	first := values[0]
	return &ydb.WizardSearchResult{
		DateForward:  first.DateForward,
		DateBackward: first.DateBackward,
		MinPrice:     m.buildMinPrice(fares),
		SearchResult: ydb.SearchResultScanner{
			Context: first.SearchResult.Context,
			Value:   m.buildSearchResult(values, fares),
		},
		FilterState: first.FilterState,
	}
}

func (m *MergerByConversion) buildMinPrice(fares []*wizard.Fare) int32 {
	var minPrice = int32(math.MaxInt32)
	for _, fare := range fares {
		price := int32(fare.GetTariff().GetValue())
		if price == 0 {
			continue
		}
		if price < minPrice {
			minPrice = price
		}
	}
	return minPrice
}

func (m *MergerByConversion) buildSearchResult(values []*ydb.WizardSearchResult, fares []*wizard.Fare) *wizard.SearchResult {
	return &wizard.SearchResult{
		Qid:           values[0].SearchResult.Value.Qid,
		Flights:       m.buildFlights(values),
		Fares:         fares,
		Version:       values[0].SearchResult.Value.Version,
		OffersCount:   ptr.Int32(int32(len(fares))),
		PollingStatus: values[0].SearchResult.Value.PollingStatus,
	}
}

func (m *MergerByConversion) buildFlights(values []*ydb.WizardSearchResult) map[string]*wizard.Flight {
	flights := make(map[string]*wizard.Flight)
	for _, value := range values {
		for route, flight := range value.SearchResult.Value.GetFlights() {
			flights[route] = flight
		}
	}
	return flights
}
