package query

import (
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
)

type Record struct {
	MainReqID string `json:"main_reqid"`
	ReqID     string `json:"reqid"`
	JobID     string `json:"job_id"`

	GeoID      int               `json:"geo_id"`
	FromGeoID  *int              `json:"from_geo_id"`
	ToGeoID    *int              `json:"to_geo_id"`
	FromID     *string           `json:"from_id"`
	ToID       *string           `json:"to_id"`
	Tld        string            `json:"tld"`
	Lang       string            `json:"lang"`
	SerpUUID   string            `json:"serp_uuid"`
	QueryText  string            `json:"query_text"`
	Device     string            `json:"device"`
	YandexUID  string            `json:"yandexuid"`
	PassportID string            `json:"passport_id"`
	Flags      map[string]string `json:"flags"`
	Organic    bool              `json:"organic"`

	Unixtime       int64           `json:"unixtime"`
	Dev            int             `json:"dev"`
	Page           string          `json:"page"`
	FromText       string          `json:"from_text"`
	ToText         string          `json:"to_text"`
	CompanyID      *int            `json:"company_id"`
	PartnerCode    *string         `json:"partner_code"`
	DirectFlight   bool            `json:"direct_flight"`
	ReturnDate     *string         `json:"return_date"`
	DepartureDate  *string         `json:"departure_date"`
	Context        dynamic.Context `json:"context"`
	FlightNumber   *string         `json:"flight_number"`
	RawDate        *string         `json:"raw_date"`
	AfterPreloader bool            `json:"after_preloader"`
	PassengersInfo string          `json:"passengers_info"`
	AviaDynamic    bool            `json:"avia_dynamic"`
	Icookie        string          `json:"icookie"`
	PP             *int            `json:"pp"`
	RawArgs        url.Values      `json:"raw_args"`
}

func NewRecord(
	queryParameters *parameters.QueryParameters,
	page consts.PageType,
	normalizedFlightNumber string,
) *Record {
	var dev int
	if queryParameters.Dev {
		dev = 1
	}
	var pp *int
	switch queryParameters.Device {
	case "desktop":
		pp = ptr.Int(502)
	case "touch":
		pp = ptr.Int(508)
	case "tablet":
		pp = ptr.Int(509)
	}

	return &Record{
		MainReqID: queryParameters.MainReqID,
		ReqID:     queryParameters.ReqID,
		JobID:     queryParameters.JobID,

		GeoID:          queryParameters.GeoID,
		FromGeoID:      queryParameters.FromGeoID,
		ToGeoID:        queryParameters.ToGeoID,
		FromID:         queryParameters.FromID,
		FromText:       queryParameters.FromText,
		ToID:           queryParameters.ToID,
		ToText:         queryParameters.ToText,
		CompanyID:      queryParameters.CompanyID,
		PartnerCode:    queryParameters.PartnerCode,
		DirectFlight:   queryParameters.DirectFlight,
		DepartureDate:  formatDate(queryParameters.DepartureDate),
		ReturnDate:     formatDate(queryParameters.ReturnDate),
		Tld:            queryParameters.Tld,
		Lang:           queryParameters.Lang.String(),
		Context:        queryParameters.Context(),
		RawDate:        nil,
		AfterPreloader: false,
		PassengersInfo: queryParameters.Passengers().ToString(),
		AviaDynamic:    queryParameters.IsDynamic(),
		SerpUUID:       queryParameters.SerpUUID,
		FlightNumber:   ptr.String(normalizedFlightNumber),
		QueryText:      strings.ToLower(queryParameters.UserRequest),
		Device:         queryParameters.Device,
		YandexUID:      queryParameters.YandexUID,
		PassportID:     queryParameters.PassportID,
		Flags:          queryParameters.Flags,
		Icookie:        queryParameters.ICookieDecrypted,
		PP:             pp,
		RawArgs:        queryParameters.RawArgs,

		Unixtime: time.Now().UTC().Unix(),
		Dev:      dev,
		Page:     page.String(),
	}
}

func formatDate(date *time.Time) *string {
	if helpers.IsNil(date) {
		return nil
	}
	return ptr.String(date.Format("2006-01-02"))
}
