package show

import (
	"math"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
)

type (
	Record struct {
		MainReqID        string                `json:"main_reqid"`
		ReqID            string                `json:"reqid"`
		JobID            string                `json:"job_id"`
		Unixtime         int64                 `json:"unixtime"`
		CarryonInfo      carryonInfo           `json:"carryon_info"`
		BaggageInfo      responses.Checked     `json:"baggage_info"`
		Qid              string                `json:"qid"`
		BaggageType      string                `json:"baggage_type"`
		TransferCount    int                   `json:"transfer_count"`
		Tariff           responses.TariffValue `json:"tariff"`
		Popularity       int                   `json:"popularity"`
		FromAviacompany  bool                  `json:"from_aviacompany"`
		ShowID           string                `json:"show_id"`
		Partner          string                `json:"partner"`
		Index            int                   `json:"index"`
		ForwardSegments  []*segment            `json:"forward_segments"`
		BackwardSegments []*segment            `json:"backward_segments"`
		ShownDate        string                `json:"shown_date"`
	}

	carryonInfo struct {
		Included   *bool      `json:"included"`
		Weight     *float64   `json:"weight"`
		Dimensions dimensions `json:"dimensions"`
	}

	dimensions struct {
		Length *int `json:"length"`
		Width  *int `json:"width"`
		Height *int `json:"height"`
		Sum    *int `json:"sum"`
	}

	segment struct {
		Number      string `json:"number"`
		DepIata     string `json:"dep_iata"`
		DepPointKey string `json:"dep_point_key"`
	}
)

func NewRecord(
	queryParameters *parameters.QueryParameters,
	variant *responses.Variant,
	index int,
	baggageType models.BaggageType,
) *Record {
	return &Record{
		MainReqID:     queryParameters.MainReqID,
		ReqID:         queryParameters.ReqID,
		JobID:         queryParameters.JobID,
		ShowID:        variant.ShowID,
		Unixtime:      time.Now().UTC().Unix(),
		CarryonInfo:   buildCarryonInfo(variant.Baggage.CarryOn),
		BaggageInfo:   variant.Baggage.Checked,
		Qid:           variant.Qid,
		BaggageType:   baggageType.String(),
		TransferCount: transferCount(variant.Forward) + transferCount(variant.Backward),
		Tariff: responses.TariffValue{
			Currency: variant.Tariff.Currency,
			Value:    variant.Tariff.Price,
		},
		Popularity:       int(math.Max(0, float64(variant.Popularity))),
		FromAviacompany:  variant.OfferFromAviacompany,
		Partner:          variant.PartnerCode,
		Index:            index,
		ForwardSegments:  dumpSegments(variant.Forward.Flights),
		BackwardSegments: dumpSegments(variant.Backward.Flights),
		ShownDate:        getShownDate(variant.Forward.Flights),
	}
}

func getShownDate(flights []*responses.PPFlight) string {
	return flights[0].DepartsAt[:10]
}

func dumpSegments(flights []*responses.PPFlight) []*segment {
	res := []*segment{}
	for _, flight := range flights {
		res = append(res, &segment{
			Number:      flight.Number,
			DepIata:     flight.Departure.Station.IATA,
			DepPointKey: flight.Departure.Station.PointKey,
		})
	}
	return res
}

func transferCount(trip *responses.Trip) int {
	if helpers.IsNil(trip) || helpers.IsNil(trip.Flights) || len(trip.Flights) == 0 {
		return 0
	}
	return len(trip.Flights) - 1
}

func buildCarryonInfo(carryon *responses.CarryOn) carryonInfo {
	return carryonInfo{
		Included: carryon.Included,
		Weight:   carryon.Weight,
		Dimensions: dimensions{
			Length: carryon.Length,
			Width:  carryon.Width,
			Height: carryon.Height,
			Sum:    carryon.DimensionsSum,
		},
	}
}
