package metrics

import (
	"fmt"
	"strings"

	libMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type WizardMetrics struct {
	*metrics.AppMetrics

	RequestTimer libMetrics.Timer

	TicketDaemonAPIInitSearchTimer libMetrics.Timer

	YdbStaticWithoutDateRequestTimer             libMetrics.Timer
	YdbStaticWithDateRequestTimer                libMetrics.Timer
	YdbExperimentalStaticWithoutDateRequestTimer libMetrics.Timer
	YdbExperimentalStaticWithDateRequestTimer    libMetrics.Timer
	YdbExperimentalStaticParallelRequestsTimer   libMetrics.Timer
	YdbStaticByPartnerWithDateRequestTimer       libMetrics.Timer
	YdbStaticByPartnerWithoutDateRequestTimer    libMetrics.Timer
	YdbDynamicRequestTimer                       libMetrics.Timer
	YdbExperimentalDynamicRequestTimer           libMetrics.Timer
	YdbDynamicByPartnerRequestTimer              libMetrics.Timer
	YdbStatementPreparationTimer                 libMetrics.Timer
	YdbStatementExecutionTimer                   libMetrics.Timer

	PersonalizationRequestTimer libMetrics.Timer

	FaresSortingTimer            libMetrics.Timer
	VariantsBuildingTimer        libMetrics.Timer
	ResponseBuildingTimer        libMetrics.Timer
	FaresProtobufDeserialization libMetrics.Timer

	SharedFlightsMultiGoodRequestTimer libMetrics.Timer
	SharedFlightsMultiBadRequestTimer  libMetrics.Timer
	SharedFlightsGoodRequestTimer      libMetrics.Timer
	SharedFlightsBadRequestTimer       libMetrics.Timer
}

func createRatedHistogram(registry libMetrics.Registry, name string, durationBuckets libMetrics.DurationBuckets) libMetrics.Timer {
	requestTimingsHistogram := registry.DurationHistogram(
		name,
		durationBuckets,
	)
	solomon.Rated(requestTimingsHistogram)
	return requestTimingsHistogram
}

func NewWizardMetrics(appMetricsRegistry *solomon.Registry) *WizardMetrics {
	timingsMetrics := appMetricsRegistry.WithPrefix(timingsPrefix)
	tdAPIMetrics := appMetricsRegistry.WithPrefix(ticketDaemonAPIPrefix)
	ydbMetrics := appMetricsRegistry.WithPrefix(ydbPrefix)
	personalizationMetrics := appMetricsRegistry.WithPrefix(personalizationPrefix)
	sharedFlightsMetrics := appMetricsRegistry.WithPrefix(sharedFlightsPrefix)
	return &WizardMetrics{
		AppMetrics: metrics.NewAppMetrics(appMetricsRegistry),

		RequestTimer: createRatedHistogram(timingsMetrics, wizardRequestTimings.String(), DefaultTimingsBuckets),

		TicketDaemonAPIInitSearchTimer: createRatedHistogram(tdAPIMetrics, ticketDaemonAPIInitSearchTimings.String(), DefaultTimingsBuckets),

		YdbStaticWithoutDateRequestTimer:             createRatedHistogram(ydbMetrics, ydbStaticWithoutDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbStaticByPartnerWithoutDateRequestTimer:    createRatedHistogram(ydbMetrics, ydbStaticByPartnerWithoutDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbExperimentalStaticWithoutDateRequestTimer: createRatedHistogram(ydbMetrics, ydbExperimentalStaticWithoutDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbStaticWithDateRequestTimer:                createRatedHistogram(ydbMetrics, ydbStaticWithDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbStaticByPartnerWithDateRequestTimer:       createRatedHistogram(ydbMetrics, ydbStaticByPartnerWithDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbExperimentalStaticWithDateRequestTimer:    createRatedHistogram(ydbMetrics, ydbExperimentalStaticWithDateRequestTimings.String(), DefaultTimingsBuckets),
		YdbExperimentalStaticParallelRequestsTimer:   createRatedHistogram(ydbMetrics, ydbExperimentalStaticParallelRequestsTimings.String(), DefaultTimingsBuckets),
		YdbDynamicRequestTimer:                       createRatedHistogram(ydbMetrics, ydbDynamicRequestTimings.String(), DefaultTimingsBuckets),
		YdbDynamicByPartnerRequestTimer:              createRatedHistogram(ydbMetrics, ydbDynamicByPartnerRequestTimings.String(), DefaultTimingsBuckets),
		YdbExperimentalDynamicRequestTimer:           createRatedHistogram(ydbMetrics, ydbExperimentalDynamicRequestTimings.String(), DefaultTimingsBuckets),
		YdbStatementPreparationTimer:                 createRatedHistogram(ydbMetrics, ydbStatementPreparationTimings.String(), DefaultTimingsBuckets),
		YdbStatementExecutionTimer:                   createRatedHistogram(ydbMetrics, ydbStatementExecutionTimings.String(), DefaultTimingsBuckets),

		PersonalizationRequestTimer: createRatedHistogram(personalizationMetrics, personalizationRequestTimings.String(), DefaultTimingsBuckets),

		FaresSortingTimer:            createRatedHistogram(timingsMetrics, faresSortingTimings.String(), DefaultTimingsBuckets),
		VariantsBuildingTimer:        createRatedHistogram(timingsMetrics, variantsBuildingTimings.String(), DefaultTimingsBuckets),
		ResponseBuildingTimer:        createRatedHistogram(timingsMetrics, responseBuildingTimings.String(), DefaultTimingsBuckets),
		FaresProtobufDeserialization: createRatedHistogram(timingsMetrics, faresProtobufDeserialization.String(), DefaultTimingsBuckets),

		SharedFlightsGoodRequestTimer:      createRatedHistogram(sharedFlightsMetrics, flightGoodRequestTimings.String(), DefaultTimingsBuckets),
		SharedFlightsBadRequestTimer:       createRatedHistogram(sharedFlightsMetrics, flightBadRequestTimings.String(), DefaultTimingsBuckets),
		SharedFlightsMultiGoodRequestTimer: createRatedHistogram(sharedFlightsMetrics, flightMultiGoodRequestTimings.String(), DefaultTimingsBuckets),
		SharedFlightsMultiBadRequestTimer:  createRatedHistogram(sharedFlightsMetrics, flightMultiBadRequestTimings.String(), DefaultTimingsBuckets),
	}
}

func (wm *WizardMetrics) GetResponseTypeCounter(status string) libMetrics.Counter {
	return wm.GetOrCreateCounter(wizardResponseTypes, nil, status)
}

func (wm *WizardMetrics) GetReqIDCounter(reqIDGroup, reqIDType string) libMetrics.Counter {
	return wm.GetOrCreateCounter(reqidTypes, nil, fmt.Sprintf("%s.%s", reqIDGroup, reqIDType))
}

func (wm *WizardMetrics) GetErrorCounter(errorType string) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("errors.%s", errorType))
}

func (wm *WizardMetrics) GetPrecacheErrorCounter(cacheName string) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("cache_update_error.%s", cacheName))
}

func (wm *WizardMetrics) GetStaticRequestTimer() libMetrics.Timer {
	return wm.GetOrCreateHistogram(timingsPrefix, nil, "static", DefaultTimingsBuckets)
}

func (wm *WizardMetrics) GetDynamicRequestTimer() libMetrics.Timer {
	return wm.GetOrCreateHistogram(timingsPrefix, nil, "dynamic", DefaultTimingsBuckets)
}

func (wm *WizardMetrics) GetPointToPointRequestTimer() libMetrics.Timer {
	return wm.GetOrCreateHistogram(timingsPrefix, nil, "point_to_point", DefaultTimingsBuckets)
}

func (wm *WizardMetrics) GetFlightRequestTimer() libMetrics.Timer {
	return wm.GetOrCreateHistogram(timingsPrefix, nil, "flight", DefaultTimingsBuckets)
}

func (wm *WizardMetrics) GetCommonRequestTimer() libMetrics.Timer {
	return wm.GetOrCreateHistogram(timingsPrefix, nil, "common", DefaultTimingsBuckets)
}

func buildPpResponseType(isDynamic, isYak bool) string {
	resultType := strings.Builder{}
	if isDynamic {
		resultType.WriteString("dynamic_")
	} else {
		resultType.WriteString("static_")
	}
	if isYak {
		resultType.WriteString("yak")
	} else {
		resultType.WriteString("ok")
	}
	return resultType.String()
}

func buildRequestType(isDynamic bool) string {
	if isDynamic {
		return "dynamic"
	}
	return "static"
}

func (wm *WizardMetrics) GetCacheMissCounter(isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("cache_misses.all.%s", buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetCacheMissByNationalVersionCounter(nationalVersion string, isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("cache_misses.%s.%s", nationalVersion, buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetExperimentalCacheMissCounter(isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("cache_misses.exp.all.%s", buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetExperimentalCacheMissByNationalVersionCounter(nationalVersion string, isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("cache_misses.exp.%s.%s", nationalVersion, buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetYakCounter(isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("yaks.all.%s", buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetYakCounterByNationalVersion(nationalVersion string, isDynamic bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("yaks.%s.%s", nationalVersion, buildRequestType(isDynamic)))
}

func (wm *WizardMetrics) GetPPResponseTypeCounter(isDynamic, isYak bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("pp_response.all.%s", buildPpResponseType(isDynamic, isYak)))
}

func (wm *WizardMetrics) GetPPResponseTypeByNationalVersionCounter(nationalVersion string, isDynamic, isYak bool) libMetrics.Counter {
	return wm.GetOrCreateCounter(general, nil, fmt.Sprintf("pp_response.%s.%s", nationalVersion, buildPpResponseType(isDynamic, isYak)))
}

func (wm *WizardMetrics) GetSharedFlightsHTTPCodeCounter(statusCode int) libMetrics.Counter {
	return wm.GetOrCreateCounter(sharedFlightsPrefix, nil, fmt.Sprintf("code.%d", statusCode))
}

func (wm *WizardMetrics) GetPersonalizationCacheHitsCounter() libMetrics.Counter {
	return wm.GetOrCreateCounter(personalizationPrefix, nil, "cache_hit")
}

func (wm *WizardMetrics) GetPersonalizationActivationCounter() libMetrics.Counter {
	return wm.GetOrCreateCounter(personalizationPrefix, nil, "activation")
}

func createRatedCounter(appMetricsRegistry libMetrics.Registry, name string) libMetrics.Counter {
	counter := appMetricsRegistry.Counter(name)
	solomon.Rated(counter)
	return counter
}

var globalWizardMetrics *WizardMetrics

func SetGlobalWizardMetrics(wizardMetrics *WizardMetrics) {
	globalWizardMetrics = wizardMetrics
}

func GlobalWizardMetrics() *WizardMetrics {
	if helpers.IsNil(globalWizardMetrics) {
		globalWizardMetrics = NewWizardMetrics(solomon.NewRegistry(solomon.NewRegistryOpts()))
	}
	return globalWizardMetrics
}
