package middlewares

import (
	"fmt"
	"net/http"
	"strings"

	"a.yandex-team.ru/travel/avia/library/go/utils"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/responses"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
)

type ReqIDMetricsMiddleware struct{}

func getReqIDType(reqID string) string {
	if len(reqID) == 0 {
		return "empty"
	}
	parts := strings.Split(reqID, "-")
	if len(parts) <= 4 {
		return "common"
	}
	if helpers.IsDigit(parts[len(parts)-1]) {
		return "numeric"
	}
	return parts[len(parts)-1]
}

var (
	bannedReqIDsTypes  = containers.NewSetOfString("empty")
	ignoredReqIDsTypes = containers.NewSetOfString("XML")
)

func (m *ReqIDMetricsMiddleware) Handle(next http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		queryParameters := utils.HTTP.ParseQueryParameters(r)
		mainReqIDValue := queryParameters.Get("main_reqid")
		reqIDValue := queryParameters.Get("reqid")
		reqIDType := getReqIDType(reqIDValue)
		mainReqIDType := getReqIDType(mainReqIDValue)
		wizardMetrics := metrics.GlobalWizardMetrics()
		go wizardMetrics.GetReqIDCounter("main_reqid", mainReqIDType).Inc()
		go wizardMetrics.GetReqIDCounter("reqid", reqIDType).Inc()

		if bannedReqIDsTypes.Contains(strings.ToUpper(mainReqIDType)) {
			utils.HTTP.JSON(
				w, http.StatusBadRequest,
				responses.NewErrorResponse(fmt.Sprintf("main_reqid type %s is banned", mainReqIDType), "bad_arguments"),
			)
			return
		}
		if bannedReqIDsTypes.Contains(strings.ToUpper(reqIDType)) {
			utils.HTTP.JSON(
				w, http.StatusBadRequest,
				responses.NewErrorResponse(fmt.Sprintf("reqid type %s is banned", reqIDType), "bad_arguments"),
			)
			return
		}

		if ignoredReqIDsTypes.Contains(strings.ToUpper(mainReqIDType)) {
			utils.HTTP.JSON(
				w, http.StatusNoContent,
				responses.NewErrorResponse(fmt.Sprintf("main_reqid type %s is ignored", mainReqIDType), "bad_arguments"),
			)
			return
		}
		if ignoredReqIDsTypes.Contains(strings.ToUpper(reqIDType)) {
			utils.HTTP.JSON(
				w, http.StatusNoContent,
				responses.NewErrorResponse(fmt.Sprintf("reqid type %s is ignored", reqIDType), "bad_arguments"),
			)
			return
		}
		next.ServeHTTP(w, r)
	})
}

func NewReqIDMetricsMiddleware() *ReqIDMetricsMiddleware {
	return &ReqIDMetricsMiddleware{}
}
