package middlewares

import (
	"net/http"

	"go.uber.org/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers/props"
)

type TooManyRequestsMiddleware struct {
	logger              log.Logger
	inflightRequests    *atomic.Uint32
	maxInflightRequests uint32
}

func (m *TooManyRequestsMiddleware) Handle(next http.Handler) http.Handler {
	return http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			r = r.WithContext(props.WithSearchPropsStore(r.Context(), m.logger))
			m.inflightRequests.Inc()
			defer m.inflightRequests.Dec()
			if m.inflightRequests.Load() > m.maxInflightRequests {
				w.WriteHeader(http.StatusTooManyRequests)
				_, _ = w.Write([]byte("Too many requests"))
				return
			}
			next.ServeHTTP(w, r)
		},
	)
}

func NewTooManyRequestsMiddleware(logger log.Logger, maxInflightRequests uint32) *TooManyRequestsMiddleware {
	return &TooManyRequestsMiddleware{
		logger:              logger,
		maxInflightRequests: maxInflightRequests,
		inflightRequests:    atomic.NewUint32(0),
	}
}
