package repositories

import (
	"fmt"
	"strconv"
	"strings"
	"sync"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
)

type CachedCheckedBaggage interface {
	ParseBaggageCode(BaggageCode string) (baggage *models.CheckedBaggage, err error)
}

type CachedCheckedBaggageRepository struct {
	cache sync.Map
}

func NewCachedBaggageCodeRepository() CachedCheckedBaggage {
	return &CachedCheckedBaggageRepository{}
}

func (repository *CachedCheckedBaggageRepository) ParseBaggageCode(code string) (baggage *models.CheckedBaggage, err error) {
	if checkedBaggage, ok := repository.cache.Load(code); ok {
		if !helpers.IsNil(checkedBaggage) {
			return checkedBaggage.(*models.CheckedBaggage), nil
		}
		return nil, domain.NewWizardError(fmt.Sprintf("bad baggage code format: %s", code), domain.BadBaggageCode)
	}
	processedCode := strings.ReplaceAll(code, "N", "p")
	processedCode = strings.ReplaceAll(processedCode, "d", "p")
	parts := strings.Split(processedCode, "p")
	if len(parts) < 3 {
		repository.cache.Store(code, nil)
		return nil, domain.NewWizardError(fmt.Sprintf("bad baggage code format: %s", code), domain.BadBaggageCode)
	}
	var included, pieces, weight *int
	if parsed, err := strconv.Atoi(parts[0]); err == nil {
		included = &parsed
	}
	if parsed, err := strconv.Atoi(parts[1]); err == nil {
		pieces = &parsed
	}
	if parsed, err := strconv.Atoi(parts[2]); err == nil {
		weight = &parsed
	}
	if included == nil || *included == 0 {
		baggage := &models.CheckedBaggage{Included: included}
		repository.cache.Store(code, baggage)
		return baggage, nil
	}
	baggage = &models.CheckedBaggage{
		Included: included,
		Pieces:   pieces,
		Weight:   weight,
	}
	repository.cache.Store(code, baggage)
	return baggage, nil
}
