package repositories

import (
	"strings"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type (
	Company interface {
		GetAll() []*models.Company
		GetByID(id int) (*models.Company, bool)
		GetLogoByID(id int) string
		GetByIata(code string) (*models.Company, bool)
		GetBySirena(code string) (*models.Company, bool)
		GetByCode(code string) *models.Company
	}

	idToCompanyMapper     map[int]*models.Company
	iataToCompanyMapper   map[string]*models.Company
	sirenaToCompanyMapper map[string]*models.Company

	CompanyRepository struct {
		companyReference      *references.Company
		idToCompanyMapper     idToCompanyMapper
		iataToCompanyMapper   iataToCompanyMapper
		sirenaToCompanyMapper sirenaToCompanyMapper
	}
)

const fakeCompanyWithLogoID = 57942

func NewCompanyRepository(
	aviaCompanyReference *references.AviaCompany,
	companyReference *references.Company,
) Company {
	repository := &CompanyRepository{
		companyReference:      companyReference,
		idToCompanyMapper:     make(idToCompanyMapper),
		iataToCompanyMapper:   make(iataToCompanyMapper),
		sirenaToCompanyMapper: make(sirenaToCompanyMapper),
	}

	for _, company := range companyReference.GetAll() {
		repository.idToCompanyMapper[company.ID] = company

		if company.Iata != "" {
			repository.iataToCompanyMapper[strings.ToLower(company.Iata)] = company
		}

		if company.SirenaID != "" {
			repository.sirenaToCompanyMapper[strings.ToLower(company.SirenaID)] = company
		}
	}

	for _, aviaCompany := range aviaCompanyReference.GetAll() {
		if company, found := repository.idToCompanyMapper[aviaCompany.RaspCompanyID]; found {
			if company.Iata != "" {
				repository.iataToCompanyMapper[strings.ToLower(company.Iata)] = company
			}
		}
	}
	return repository
}

func (repository *CompanyRepository) GetAll() []*models.Company {
	return repository.companyReference.GetAll()
}

func (repository *CompanyRepository) GetByID(id int) (*models.Company, bool) {
	company, found := repository.idToCompanyMapper[id]
	return company, found
}

func (repository *CompanyRepository) GetLogoByID(id int) string {
	if company, found := repository.idToCompanyMapper[id]; found {
		return company.SvgLogo2
	}
	return repository.idToCompanyMapper[fakeCompanyWithLogoID].SvgLogo2
}

func (repository *CompanyRepository) GetByIata(code string) (*models.Company, bool) {
	company, found := repository.iataToCompanyMapper[strings.ToLower(code)]
	return company, found
}

func (repository *CompanyRepository) GetBySirena(code string) (*models.Company, bool) {
	company, found := repository.sirenaToCompanyMapper[strings.ToLower(code)]
	return company, found
}

func (repository *CompanyRepository) GetByCode(code string) *models.Company {
	if company, found := repository.GetByIata(code); found {
		return company
	}
	if company, found := repository.GetBySirena(code); found {
		return company
	}
	return nil
}
