package repositories

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

type Conversions map[string]float32

type ConversionRepository struct {
	logger            log.Logger
	settings          *settings.ConversionSettings
	partnerRepository Partner
	refs              *references.ResourceRegistry
	conversions       Conversions
}

func NewConversionRepository(
	logger log.Logger,
	settings *settings.ConversionSettings,
	partnerRepository Partner,
	refs *references.ResourceRegistry,
) *ConversionRepository {
	return &ConversionRepository{
		logger:            logger,
		settings:          settings,
		refs:              refs,
		partnerRepository: partnerRepository,
		conversions:       make(Conversions),
	}
}

func (repository *ConversionRepository) GetAll() Conversions {
	return repository.conversions
}

func (repository *ConversionRepository) Precache() error {
	repository.precache()
	go repository.repeatedPrecache()

	return nil
}

func (repository *ConversionRepository) repeatedPrecache() {
	for {
		time.Sleep(repository.settings.RecacheInterval)
		repository.precache()
	}
}

func (repository *ConversionRepository) precache() {
	conversions := repository.refs.Conversion.GetAll()
	repository.logger.Info(fmt.Sprintf("Got conversions: %v", len(conversions)))

	partners := repository.partnerRepository.GetPartners()
	repository.logger.Info(fmt.Sprintf("Got partners: %v", len(partners)))

	res := make(Conversions, len(conversions))
	for _, conversion := range conversions {
		for _, partner := range partners {
			if partner.BillingOrderID == conversion.BillingOrderID && conversion.Source == repository.settings.SourceName {
				res[partner.Code] = conversion.Conversion
				break
			}
		}
	}
	repository.conversions = res
	repository.logger.Info(fmt.Sprintf("Update conversions cache: %v", len(res)))
}
