package repositories

import (
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type (
	Direction interface {
		GetPopularity(departureID, arrivalID int) int
		IsPossibleVariant(departureID, arrivalID int, nationalVersion string) bool
	}

	popularityKey struct {
		departureID int
		arrivalID   int
	}

	bidirectionalKey struct {
		departureID     int
		arrivalID       int
		nationalVersion string
	}

	popularityByDirection map[popularityKey]int
	possibleDirection     map[bidirectionalKey]struct{}

	DirectionRepository struct {
		directionReference *references.Direction

		popularityByDirection popularityByDirection
		possibleDirection     possibleDirection
	}
)

func NewDirectionRepository(directionReference *references.Direction) Direction {
	repository := &DirectionRepository{
		directionReference:    directionReference,
		popularityByDirection: make(popularityByDirection),
		possibleDirection:     make(possibleDirection),
	}

	for _, direction := range directionReference.GetAll() {
		if currentPopularity, ok := repository.popularityByDirection[getPopularityKey(direction)]; !ok || currentPopularity < direction.Popularity {
			repository.popularityByDirection[getPopularityKey(direction)] = direction.Popularity
		}
		repository.possibleDirection[getBidirectionalKey(
			direction.DepartureSettlementID,
			direction.ArrivalSettlementID,
			direction.NationalVersion,
		)] = struct{}{}
	}

	return repository
}

func (repository *DirectionRepository) GetPopularity(departureID, arrivalID int) int {
	key := popularityKey{
		departureID: departureID,
		arrivalID:   arrivalID,
	}
	if popularity, ok := repository.popularityByDirection[key]; ok {
		return popularity
	}
	return 0
}

func (repository *DirectionRepository) IsPossibleVariant(departureID, arrivalID int, nationalVersion string) bool {
	_, ok := repository.possibleDirection[getBidirectionalKey(departureID, arrivalID, nationalVersion)]
	return ok
}

func getBidirectionalKey(departureID, arrivalID int, nationalVersion string) bidirectionalKey {
	if departureID > arrivalID {
		arrivalID, departureID = departureID, arrivalID
	}
	return bidirectionalKey{
		departureID:     departureID,
		arrivalID:       arrivalID,
		nationalVersion: nationalVersion,
	}
}

func getPopularityKey(direction *models.Direction) popularityKey {
	return popularityKey{
		departureID: direction.DepartureSettlementID,
		arrivalID:   direction.ArrivalSettlementID,
	}
}
