package repositories

import (
	"reflect"
	"strings"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
)

type NameSearchIndex interface {
	FindExact(models.ModelType, string) []interface{}
	FindWords(models.ModelType, string) []interface{}
}

type indexNodes map[models.ModelType]*nameSearchIndexNode

type NameSearchIndexGroup struct {
	pointSynonymRepo    PointSynonym
	translatedTitleRepo TranslatedTitle

	indexNodes indexNodes
}

func NewNameSearchIndex(
	pointSynonymRepository PointSynonym,
	translatedTitleRepository TranslatedTitle,
	countryRepo Country,
	settlementRepo Settlement,
	stationRepo Station,
	regionRepo Region,
) NameSearchIndex {
	nodes := make(indexNodes)

	for _, contentType := range []models.ModelType{
		models.CountryModel,
		models.SettlementModel,
		models.StationModel,
		models.RegionModel,
	} {
		nodes[contentType] = newNameSearchIndexNode(contentType, pointSynonymRepository, translatedTitleRepository)
	}

	indexGroup := &NameSearchIndexGroup{
		pointSynonymRepo:    pointSynonymRepository,
		translatedTitleRepo: translatedTitleRepository,
		indexNodes:          nodes,
	}
	indexGroup.indexNodes[models.CountryModel].populate(convertToObjectWithTitle(countryRepo.GetAll()))
	indexGroup.indexNodes[models.SettlementModel].populate(convertToObjectWithTitle(settlementRepo.GetAll()))
	indexGroup.indexNodes[models.StationModel].populate(convertToObjectWithTitle(stationRepo.GetAll()))
	indexGroup.indexNodes[models.RegionModel].populate(convertToObjectWithTitle(regionRepo.GetAll()))

	return indexGroup
}

func convertToObjectWithTitle(slice interface{}) []models.Point {
	s := reflect.ValueOf(slice)
	if s.Kind() != reflect.Slice {
		panic("InterfaceSlice() given a non-slice type")
	}

	ret := make([]models.Point, s.Len())

	for i := 0; i < s.Len(); i++ {
		ret[i] = s.Index(i).Interface().(models.Point)
	}

	return ret
}

func (index *NameSearchIndexGroup) FindExact(contentType models.ModelType, text string) []interface{} {
	text = helpers.Normalize(text)
	words := helpers.CutOnWords(text)

	indexNode, found := index.indexNodes[contentType]
	if !found {
		return make([]interface{}, 0)
	}

	objects := make(containers.SetOfInterface)
	if otherObjectIDs, found := indexNode.exactObjectIndex[text]; found {
		objects.Extend(otherObjectIDs)
	}

	if len(words) != 1 || words[0] == text {
		if otherObjectIDs, found := indexNode.exactObjectIndex[text]; found {
			objects.Extend(otherObjectIDs)
		}
	}

	return objects.ToSlice()
}

func (index *NameSearchIndexGroup) FindWords(contentType models.ModelType, text string) []interface{} {
	words := helpers.CutOnWords(helpers.Normalize(text))
	text = strings.Join(words, " ")
	result := make([]interface{}, 0)

	indexNode, found := index.indexNodes[contentType]
	if found {
		if objects, found := indexNode.wordsObjectIndex[text]; found {
			return objects.ToSlice()
		}
	}

	return result
}

type nameSearchIndexNode struct {
	contentType         models.ModelType
	pointSynonymRepo    PointSynonym
	translatedTitleRepo TranslatedTitle

	exactObjectIndex containers.SetOfInterfaceByText
	wordsObjectIndex containers.SetOfInterfaceByText
}

func newNameSearchIndexNode(
	contentType models.ModelType,
	pointSynonymRepo PointSynonym,
	translatedTitleRepo TranslatedTitle,
) *nameSearchIndexNode {
	return &nameSearchIndexNode{
		contentType:         contentType,
		pointSynonymRepo:    pointSynonymRepo,
		translatedTitleRepo: translatedTitleRepo,
		exactObjectIndex:    make(containers.SetOfInterfaceByText),
		wordsObjectIndex:    make(containers.SetOfInterfaceByText),
	}
}

func (index *nameSearchIndexNode) populate(objects []models.Point) {
	for _, object := range objects {
		index.populateSynonymObject(object)
		index.populateTranslation(object)
	}
}

func (index *nameSearchIndexNode) populateSynonymObject(object models.Point) {
	synonyms, found := index.pointSynonymRepo.GetByObjectID(object.GetID())
	if !found {
		return
	}

	for _, synonym := range synonyms {
		if synonym.ContentTypeID != int(index.contentType) {
			return
		}

		text := helpers.Normalize(synonym.Title)
		if text != "" {
			index.exactObjectIndex.EnsureKey(text)
			index.exactObjectIndex[text].Add(object)
		}
	}
}

func (index *nameSearchIndexNode) populateTranslation(object models.Point) {
	titleModel, found := index.translatedTitleRepo.GetByID(object.GetTitleID())
	if !found {
		return
	}

	for _, title := range []string{
		titleModel.RuNominative,
		titleModel.EnNominative,
		titleModel.TrNominative,
		titleModel.UkNominative,
		titleModel.DeNominative,
	} {
		index.populateObjectByText(title, object)
	}

}

func (index *nameSearchIndexNode) populateObjectByText(
	text string,
	object models.Point,
) {
	text = helpers.Normalize(text)

	if text == "" {
		return
	}

	index.exactObjectIndex.EnsureKey(text)
	index.exactObjectIndex[text].Add(object)

	words := helpers.CutOnWords(text)
	lenWords := len(words)
	if lenWords == 1 {
		return
	}

	for i := 0; i < lenWords; i++ {
		for j := i; j < lenWords+1; j++ {
			wordCombination := strings.Join(words[i:j], " ")
			if len(wordCombination) > 0 && wordCombination != text {
				index.wordsObjectIndex.EnsureKey(wordCombination)
				index.wordsObjectIndex[wordCombination].Add(object)
			}
		}
	}
}
