package repositories

import (
	"sync/atomic"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
)

type (
	Partner interface {
		GetDisabledPartnersCodes(nationalVersion string) containers.SetOfString
		GetPartnerByCode(code string) (*models.Partner, bool)
		IsEnabled(partnerCode, nationalVersion string) bool
		GetPartners() map[string]*models.Partner
	}

	idToPartnerMapper   map[int]*models.Partner
	codeToPartnerMapper map[string]*models.Partner

	PartnerRepository struct {
		partnerReference *references.Partners
		updateInterval   time.Duration

		idToPartnerMapper   atomic.Value
		codeToPartnerMapper atomic.Value
		partners            atomic.Value
	}
)

func NewPartnerRepository(partnerReference *references.Partners, updateInterval time.Duration) Partner {
	repository := &PartnerRepository{
		partnerReference: partnerReference,
		updateInterval:   updateInterval,
	}
	repository.update()
	go repository.updatePeriodically()
	return repository
}

func (partnerRepository *PartnerRepository) update() {
	updatedPartners := partnerRepository.partnerReference.GetAll()
	partnerRepository.partners.Store(updatedPartners)

	idToPartnerMapper := make(idToPartnerMapper)
	codeToPartnerMapper := make(codeToPartnerMapper)

	for _, partner := range updatedPartners {
		idToPartnerMapper[partner.ID] = partner
		codeToPartnerMapper[partner.Code] = partner
	}
	partnerRepository.idToPartnerMapper.Store(idToPartnerMapper)
	partnerRepository.codeToPartnerMapper.Store(codeToPartnerMapper)
}

func (partnerRepository *PartnerRepository) updatePeriodically() {
	for range time.Tick(partnerRepository.updateInterval) {
		partnerRepository.update()
	}
}

func (partnerRepository *PartnerRepository) GetDisabledPartnersCodes(nationalVersion string) containers.SetOfString {
	result := containers.SetOfString{}
	for _, p := range partnerRepository.partners.Load().([]*models.Partner) {
		if p.Enabled && p.EnabledInWizard[nationalVersion] {
			result.Add(p.Code)
		}
	}
	return result
}

func (partnerRepository *PartnerRepository) IsEnabled(partnerCode, nationalVersion string) bool {
	if partner, exists := partnerRepository.codeToPartnerMapper.Load().(codeToPartnerMapper)[partnerCode]; exists {
		return partner.Enabled && partner.EnabledInWizard[nationalVersion]
	}
	return true
}

func (partnerRepository *PartnerRepository) GetPartnerByCode(code string) (*models.Partner, bool) {
	partner, found := partnerRepository.codeToPartnerMapper.Load().(codeToPartnerMapper)[code]
	return partner, found
}

func (partnerRepository *PartnerRepository) GetPartners() map[string]*models.Partner {
	return partnerRepository.codeToPartnerMapper.Load().(codeToPartnerMapper)
}
