package repositories

import (
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type PopularSettlements interface {
	GetPopularByRegion(nationalVersion string, countryID int) (*models.Settlement, bool)
	GetPopularByCountry(nationalVersion string, countryID int) (*models.Settlement, bool)
}

type PopularSettlementsRepository struct {
	popularByCountry popularSettlementInGeoObjectByNV
	popularByRegion  popularSettlementInGeoObjectByNV
}

type popularSettlementInGeoObjectByNV map[string]map[int]*models.Settlement

func (m popularSettlementInGeoObjectByNV) ensureNationalVersion(nv string) {
	if _, ok := m[nv]; !ok {
		m[nv] = make(map[int]*models.Settlement)
	}
}

func NewPopularSettlementsRepository(
	settlementPopularities *references.SettlementPopularities,
	settlementsCache *references.Settlement,
) *PopularSettlementsRepository {
	popularByRegion := make(popularSettlementInGeoObjectByNV)
	popularByCountry := make(popularSettlementInGeoObjectByNV)

	for _, popularity := range settlementPopularities.GetOrderedPopularities() {
		nv := popularity.NationalVersion
		s, ok := settlementsCache.GetByID(popularity.SettlementID)
		if !ok {
			continue
		}

		popularByRegion.ensureNationalVersion(nv)
		popularByRegion[nv][s.RegionID] = s

		popularByCountry.ensureNationalVersion(nv)
		popularByCountry[nv][s.CountryID] = s
	}
	return &PopularSettlementsRepository{popularByCountry: popularByCountry, popularByRegion: popularByRegion}
}

func (repository *PopularSettlementsRepository) GetPopularByRegion(
	nationalVersion string,
	regionID int,
) (*models.Settlement, bool) {
	if _, ok := repository.popularByRegion[nationalVersion]; !ok {
		return nil, false
	}
	s, ok := repository.popularByRegion[nationalVersion][regionID]
	return s, ok
}

func (repository *PopularSettlementsRepository) GetPopularByCountry(
	nationalVersion string,
	countryID int,
) (*models.Settlement, bool) {
	if _, ok := repository.popularByCountry[nationalVersion]; !ok {
		return nil, false
	}
	s, ok := repository.popularByCountry[nationalVersion][countryID]
	return s, ok
}
