package repositories

import (
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/opentracing/opentracing-go"

	ydbLib "a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	ydbResult "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories/ydb"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

type ResultsRepository interface {
	GetStatic(query *StaticQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetPersonalizedStatic(query *StaticQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetPersonalizedStaticByPartner(query *StaticQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetExperimentalStatic(query *StaticQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetStaticByPartner(query *StaticQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetDynamic(query *DynamicQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetExperimentalDynamic(query *DynamicQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
	GetDynamicByPartners(query *DynamicQuery, ctx context.Context) ([]*ydbResult.WizardSearchResult, error)
}

type YDBResultsRepository struct {
	appLogger                                log.Logger
	sessionPool                              *table.SessionPool
	transactionControl                       *table.TransactionControl
	minPriceByDirectionQuery                 string
	minPriceByDirectionWithTTLQuery          string
	minPriceByDirectionWithTTLParallelQuery  string
	minPriceByDirectionByPartnerQuery        string
	personalizationDateForwardQuery          string
	personalizationDateForwardByPartnerQuery string
	staticWithDateQuery                      string
	staticWithDateAndTTLQuery                string
	staticWithDateByPartnerQuery             string
	dynamicQuery                             string
	dynamicWithTTLQuery                      string
	dynamicByPartnerQuery                    string
	preparedSessionsCount                    int
	retryer                                  *table.Retryer
	staticRequestTimeout                     time.Duration
	dynamicRequestTimeout                    time.Duration
	dynamicRequestByPartnerTimeout           time.Duration
	farDateTTL                               int
	nearDateTTL                              int
	nearDaysCount                            int
	fetchVariantsFromPartnerTable            bool
}

func NewYDBResultsRepository(
	appLogger log.Logger,
	sessionPool *table.SessionPool,
	ydbSettings *settings.YdbSettings,
	searchSettings *settings.SearchSettings,
) ResultsRepository {
	readOnlyTransactionControl := table.TxControl(
		table.BeginTx(table.WithStaleReadOnly()),
		table.CommitTx(),
	)
	repository := &YDBResultsRepository{
		appLogger:   appLogger,
		sessionPool: sessionPool,
		retryer: &table.Retryer{
			MaxRetries:      ydbLib.DefaultMaxRetries,
			Backoff:         ydbLib.LogBackoff{SlotDuration: time.Millisecond},
			SessionProvider: sessionPool,
			RetryChecker:    ydbLib.RetryChecker{RetryNotFound: false},
		},
		transactionControl:                       readOnlyTransactionControl,
		minPriceByDirectionQuery:                 fmt.Sprintf(minPriceByDirectionQueryTemplate, ydbSettings.Table),
		minPriceByDirectionByPartnerQuery:        fmt.Sprintf(minPriceByDirectionByPartnerQueryTemplate, ydbSettings.PartnerTable),
		minPriceByDirectionWithTTLParallelQuery:  fmt.Sprintf(minPriceByDirectionWithTTLParallelQueryTemplate, ydbSettings.ExperimentalTable),
		minPriceByDirectionWithTTLQuery:          fmt.Sprintf(minPriceByDirectionWithTTLQueryTemplate, ydbSettings.ExperimentalTable),
		staticWithDateQuery:                      fmt.Sprintf(staticWithDateQueryTemplate, ydbSettings.Table),
		personalizationDateForwardQuery:          fmt.Sprintf(personalizationDateForwardOrMinPriceByDirectionTemplate, ydbSettings.Table),
		personalizationDateForwardByPartnerQuery: fmt.Sprintf(personalizationDateForwardOrMinPriceByDirectionByPartnerTemplate, ydbSettings.PartnerTable),
		staticWithDateAndTTLQuery:                fmt.Sprintf(staticWithDateAndTTLQueryTemplate, ydbSettings.ExperimentalTable),
		staticWithDateByPartnerQuery:             fmt.Sprintf(staticWithDateByPartnerQueryTemplate, ydbSettings.PartnerTable),
		dynamicQuery:                             fmt.Sprintf(dynamicQueryTemplate, ydbSettings.Table),
		dynamicWithTTLQuery:                      fmt.Sprintf(dynamicWithTTLQueryTemplate, ydbSettings.ExperimentalTable),
		dynamicByPartnerQuery:                    fmt.Sprintf(dynamicQueryByPartnerTemplate, ydbSettings.PartnerTable),
		preparedSessionsCount:                    ydbSettings.PreparedSessionsCount,
		staticRequestTimeout:                     ydbSettings.StaticRequestTimeout,
		dynamicRequestTimeout:                    ydbSettings.DynamicRequestTimeout,
		dynamicRequestByPartnerTimeout:           ydbSettings.DynamicRequestByPartnerTimeout,
		farDateTTL:                               int(searchSettings.FarDateTTL.Seconds()),
		nearDateTTL:                              int(searchSettings.NearDateTTL.Seconds()),
		nearDaysCount:                            searchSettings.NearDaysCount,
		fetchVariantsFromPartnerTable:            searchSettings.FetchVariantsFromPartnerTable,
	}
	helpers.PanicIfNotNil(repository.fillSessionPool())
	return repository
}

const (
	upperDateForwardBound = 60

	minPriceByDirectionQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $now_in_days AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;

		$date_forward_query = (
			SELECT
				date_forward, min_price, expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $now_in_days AND
				date_forward <= $upper_date_forward_bound AND
				expires_at > $now
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$date_forward = (SELECT date_forward FROM $date_forward_query);

		SELECT
			date_forward, date_backward, search_result, filter_state, min_price
		FROM
			%[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward
		LIMIT 1;
	`

	minPriceByDirectionByPartnerQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $now_in_days AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;

		$date_forward_query = (
			SELECT
				date_forward, min_price, expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $now_in_days AND
				date_forward <= $upper_date_forward_bound AND
				expires_at > $now
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$date_forward = (SELECT date_forward FROM $date_forward_query);

		SELECT
			date_forward, date_backward, search_result, filter_state, min_price
		FROM
			%[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward
		;
	`

	minPriceByDirectionWithTTLParallelQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $ttl AS Uint64;
		DECLARE $lower_date_forward_bound AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;

		$date_forward_query = (
			SELECT
				date_forward, min_price, expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $lower_date_forward_bound AND
				date_forward <= $upper_date_forward_bound AND
				created_at > $now - $ttl
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$date_forward = (SELECT date_forward FROM $date_forward_query);

		SELECT
			date_forward, date_backward, search_result, filter_state, min_price
		FROM
			%[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward
		LIMIT 1;
	`

	minPriceByDirectionWithTTLQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $far_ttl AS Uint64;
		DECLARE $near_ttl AS Uint64;
		DECLARE $now_in_days AS Uint32;
		DECLARE $upper_near_date_forward_bound AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;

		$near_date_forward_query = (
			SELECT
				AsTuple(date_forward, min_price, expires_at) as result,
				min_price,
				expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $now_in_days AND
				date_forward <= $upper_near_date_forward_bound AND
			    created_at > $now - $near_ttl
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$far_date_forward_query = (
			SELECT
				AsTuple(date_forward, min_price, expires_at) as result,
				min_price,
				expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward > $upper_near_date_forward_bound AND
				date_forward <= $upper_date_forward_bound AND
			    created_at > $now - $far_ttl
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$near_date_result = (select result from $near_date_forward_query);
		$far_date_result = (select result from $far_date_forward_query);

		$date_forward = (
			SELECT IF (
				$near_date_result.1 is not NULL and $far_date_result.1 is not NULL,
				IF(
					$near_date_result.1 <= $far_date_result.1,
					$near_date_result.0,
					$far_date_result.0
				),
				IF (
					$near_date_result.1 is NULL,
					$far_date_result.0,
					$near_date_result.0,
				)
			)
		);

		SELECT
			date_forward, date_backward, search_result, filter_state, min_price
		FROM
			%[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward
		LIMIT 1;
	`

	personalizationDateForwardOrMinPriceByDirectionTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $now_in_days AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;
		DECLARE $date_forward AS Uint32;

		$min_price_date_forward_query = (
			SELECT
				date_forward, min_price, expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $now_in_days AND
				date_forward <= $upper_date_forward_bound AND
				expires_at > $now
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$fixed_date_forward_query = (
			SELECT
				date_forward
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward = $date_forward AND
				expires_at > $now
			LIMIT 1
		);

		$min_price_date_forward = (SELECT date_forward FROM $min_price_date_forward_query);
		$personalized_date_forward = (SELECT date_forward from $fixed_date_forward_query);
		$result_min_price = (SELECT COALESCE($personalized_date_forward, $min_price_date_forward));

		(
			SELECT
				date_forward, date_backward, search_result, filter_state, min_price
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward = $result_min_price
			LIMIT 1
		);
	`

	personalizationDateForwardOrMinPriceByDirectionByPartnerTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $now_in_days AS Uint32;
		DECLARE $upper_date_forward_bound AS Uint32;
		DECLARE $date_forward AS Uint32;

		$min_price_date_forward_query = (
			SELECT
				date_forward, min_price, expires_at
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward >= $now_in_days AND
				date_forward <= $upper_date_forward_bound AND
				expires_at > $now
			ORDER BY
				min_price, expires_at DESC
			LIMIT 1
		);

		$fixed_date_forward_query = (
			SELECT
				date_forward
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward = $date_forward AND
				expires_at > $now
			LIMIT 1
		);

		$min_price_date_forward = (SELECT date_forward FROM $min_price_date_forward_query);
		$personalized_date_forward = (SELECT date_forward from $fixed_date_forward_query);
		$result_min_price = (SELECT COALESCE($personalized_date_forward, $min_price_date_forward));

		(
			SELECT
				date_forward, date_backward, search_result, filter_state, min_price
			FROM
				%[1]s
			WHERE
				point_from = $point_from AND
				point_to = $point_to AND
				klass = $klass AND
				passengers = $passengers AND
				national_version = $national_version AND
				date_backward = 0 AND
				date_forward = $result_min_price
		);
	`

	staticWithDateQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $date_forward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward AND
			expires_at > $now
		LIMIT 1;
	`

	staticWithDateAndTTLQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $ttl AS Uint64;
		DECLARE $date_forward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward AND
			created_at > $now - $ttl
		LIMIT 1;
	`
	staticWithDateByPartnerQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $date_forward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = 0 AND
			date_forward = $date_forward AND
			expires_at > $now
		;
	`
	dynamicQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $ttl AS Uint64;
		DECLARE $date_forward AS Uint32;
		DECLARE $date_backward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = $date_backward AND
			date_forward = $date_forward AND
			expires_at > $now
		LIMIT 1;
	`

	dynamicWithTTLQueryTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $ttl AS Uint64;
		DECLARE $date_forward AS Uint32;
		DECLARE $date_backward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			date_backward = $date_backward AND
			date_forward = $date_forward AND
			created_at > $now - $ttl
		LIMIT 1;
	`
	dynamicQueryByPartnerTemplate = `
		DECLARE $point_from AS Utf8;
		DECLARE $point_to AS Utf8;
		DECLARE $klass AS Uint8;
		DECLARE $passengers AS Uint32;
		DECLARE $national_version AS Utf8;
		DECLARE $now AS Uint64;
		DECLARE $ttl AS Uint64;
		DECLARE $date_forward AS Uint32;
		DECLARE $date_backward AS Uint32;

		SELECT date_forward, date_backward, search_result, filter_state, min_price
		FROM %[1]s
		WHERE
			point_from = $point_from AND
			point_to = $point_to AND
			date_backward = $date_backward AND
			date_forward = $date_forward AND
			klass = $klass AND
			passengers = $passengers AND
			national_version = $national_version AND
			expires_at > $now;
	`
)

type StaticQuery struct {
	PointFrom       string
	PointTo         string
	Klass           uint8
	Passengers      uint32
	NationalVersion string
	NowInDays       uint32
	DepartureDate   uint32
	Now             uint64
	JobID           string
	UseTTLParallel  bool
}

type DynamicQuery struct {
	PointFrom       string
	PointTo         string
	Klass           uint8
	Passengers      uint32
	NationalVersion string
	DepartureDate   uint32
	ReturnDate      uint32
	Now             uint64
	JobID           string
}

func (repository *YDBResultsRepository) GetStatic(
	query *StaticQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB static request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("now_in_days", query.NowInDays)
	defer requestSpan.Finish()
	statement := repository.getMinPriceByDirectionQuery()
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$now_in_days", ydbLib.Uint32Value(query.NowInDays)),
		table.ValueParam("$upper_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound)),
	)
	if query.DepartureDate != 0 {
		statement = repository.staticWithDateQuery
		params.Add(table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)))
	}
	start := time.Now()
	res, err := repository.performRequest(ctx, statement, params, repository.staticRequestTimeout, query.JobID)
	if query.DepartureDate != 0 {
		metrics.GlobalWizardMetrics().YdbStaticWithDateRequestTimer.RecordDuration(time.Since(start))
	} else {
		metrics.GlobalWizardMetrics().YdbStaticWithoutDateRequestTimer.RecordDuration(time.Since(start))
	}
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) GetPersonalizedStatic(
	query *StaticQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB static request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("now_in_days", query.NowInDays)
	defer requestSpan.Finish()
	statement := repository.getMinPriceByDirectionQuery()
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$now_in_days", ydbLib.Uint32Value(query.NowInDays)),
		table.ValueParam("$upper_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound)),
	)
	if query.DepartureDate != 0 {
		statement = repository.personalizationDateForwardQuery
		params.Add(table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)))
	}
	start := time.Now()
	res, err := repository.performRequest(ctx, statement, params, repository.staticRequestTimeout, query.JobID)
	if query.DepartureDate != 0 {
		metrics.GlobalWizardMetrics().YdbStaticWithDateRequestTimer.RecordDuration(time.Since(start))
	} else {
		metrics.GlobalWizardMetrics().YdbStaticWithoutDateRequestTimer.RecordDuration(time.Since(start))
	}
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) getMinPriceByDirectionQuery() string {
	return repository.minPriceByDirectionQuery
}

func (repository *YDBResultsRepository) GetPersonalizedStaticByPartner(
	query *StaticQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB static by partner request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("now_in_days", query.NowInDays)
	defer requestSpan.Finish()
	statement := repository.minPriceByDirectionByPartnerQuery
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$now_in_days", ydbLib.Uint32Value(query.NowInDays)),
		table.ValueParam("$upper_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound)),
	)
	if query.DepartureDate != 0 {
		statement = repository.personalizationDateForwardByPartnerQuery
		params.Add(table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)))
	}
	start := time.Now()
	res, err := repository.performRequest(ctx, statement, params, repository.staticRequestTimeout, query.JobID)
	if query.DepartureDate != 0 {
		metrics.GlobalWizardMetrics().YdbStaticByPartnerWithDateRequestTimer.RecordDuration(time.Since(start))
	} else {
		metrics.GlobalWizardMetrics().YdbStaticByPartnerWithoutDateRequestTimer.RecordDuration(time.Since(start))
	}
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) GetExperimentalStatic(
	query *StaticQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB experimental static request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("now_in_days", query.NowInDays)
	defer requestSpan.Finish()
	statement := repository.minPriceByDirectionWithTTLQuery
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$now_in_days", ydbLib.Uint32Value(query.NowInDays)),
		table.ValueParam("$upper_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound)),
		table.ValueParam("$far_ttl", ydbLib.Uint64Value(uint64(repository.farDateTTL))),
		table.ValueParam("$near_ttl", ydbLib.Uint64Value(uint64(repository.nearDateTTL))),
		table.ValueParam("$upper_near_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+uint32(repository.nearDaysCount))),
	)
	if query.DepartureDate != 0 {
		statement = repository.staticWithDateAndTTLQuery
		ttl := repository.getTTLValue(query.Now, query.DepartureDate)
		params.Add(table.ValueParam("$ttl", ydbLib.Uint64Value(ttl)))
		params.Add(table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)))
	} else if query.UseTTLParallel {
		result, err := repository.performParallelRequests(ctx, query, params)
		if err != nil {
			return nil, err
		}
		return []*ydbResult.WizardSearchResult{result}, nil
	}
	start := time.Now()
	res, err := repository.performRequest(ctx, statement, params, repository.staticRequestTimeout, query.JobID)
	if query.DepartureDate != 0 {
		metrics.GlobalWizardMetrics().YdbExperimentalStaticWithDateRequestTimer.RecordDuration(time.Since(start))
	} else {
		metrics.GlobalWizardMetrics().YdbExperimentalStaticWithoutDateRequestTimer.RecordDuration(time.Since(start))
	}
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) GetStaticByPartner(
	query *StaticQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB static by partner request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("now_in_days", query.NowInDays)
	defer requestSpan.Finish()
	statement := repository.minPriceByDirectionByPartnerQuery
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$now_in_days", ydbLib.Uint32Value(query.NowInDays)),
		table.ValueParam("$upper_date_forward_bound", ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound)),
	)
	if query.DepartureDate != 0 {
		statement = repository.staticWithDateByPartnerQuery
		params.Add(table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)))
	}
	start := time.Now()
	res, err := repository.performRequest(ctx, statement, params, repository.staticRequestTimeout, query.JobID)
	if query.DepartureDate != 0 {
		metrics.GlobalWizardMetrics().YdbStaticByPartnerWithDateRequestTimer.RecordDuration(time.Since(start))
	} else {
		metrics.GlobalWizardMetrics().YdbStaticByPartnerWithoutDateRequestTimer.RecordDuration(time.Since(start))
	}
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) getTTLValue(now uint64, date uint32) uint64 {
	if uint64(date)*24*60*60-now > uint64(repository.nearDaysCount)*24*60*60 {
		return uint64(repository.farDateTTL)
	}
	return uint64(repository.nearDateTTL)
}

func (repository *YDBResultsRepository) GetDynamic(
	query *DynamicQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB dynamic request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("date_backward", query.ReturnDate)
	defer requestSpan.Finish()
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)),
		table.ValueParam("$date_backward", ydbLib.Uint32Value(query.ReturnDate)),
	)
	statement := repository.getDynamicQuery()
	start := time.Now()
	res, err := repository.performRequest(
		ctx,
		statement,
		params,
		repository.dynamicRequestTimeout,
		query.JobID,
	)
	metrics.GlobalWizardMetrics().YdbDynamicRequestTimer.RecordDuration(time.Since(start))
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) getDynamicQuery() string {
	statement := repository.dynamicQuery
	return statement
}

func (repository *YDBResultsRepository) GetExperimentalDynamic(
	query *DynamicQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB dynamic request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("date_backward", query.ReturnDate)
	defer requestSpan.Finish()
	ttl := repository.getTTLValue(query.Now, query.DepartureDate)
	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)),
		table.ValueParam("$date_backward", ydbLib.Uint32Value(query.ReturnDate)),
		table.ValueParam("$ttl", ydbLib.Uint64Value(ttl)),
	)
	statement := repository.dynamicWithTTLQuery
	start := time.Now()
	res, err := repository.performRequest(
		ctx,
		statement,
		params,
		repository.dynamicRequestTimeout,
		query.JobID,
	)
	metrics.GlobalWizardMetrics().YdbExperimentalDynamicRequestTimer.RecordDuration(time.Since(start))
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}
func (repository *YDBResultsRepository) GetDynamicByPartners(
	query *DynamicQuery,
	ctx context.Context,
) ([]*ydbResult.WizardSearchResult, error) {
	requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "YDB dynamic by partner request")
	requestSpan.SetTag("point_from", query.PointFrom)
	requestSpan.SetTag("point_to", query.PointTo)
	requestSpan.SetTag("klass", query.Klass)
	requestSpan.SetTag("passengers", query.Passengers)
	requestSpan.SetTag("national_version", query.NationalVersion)
	requestSpan.SetTag("now", query.Now)
	requestSpan.SetTag("date_forward", query.DepartureDate)
	requestSpan.SetTag("date_backward", query.ReturnDate)
	defer requestSpan.Finish()

	params := table.NewQueryParameters(
		table.ValueParam("$point_from", ydbLib.UTF8Value(query.PointFrom)),
		table.ValueParam("$point_to", ydbLib.UTF8Value(query.PointTo)),
		table.ValueParam("$klass", ydbLib.Uint8Value(query.Klass)),
		table.ValueParam("$passengers", ydbLib.Uint32Value(query.Passengers)),
		table.ValueParam("$national_version", ydbLib.UTF8Value(query.NationalVersion)),
		table.ValueParam("$now", ydbLib.Uint64Value(query.Now)),
		table.ValueParam("$date_forward", ydbLib.Uint32Value(query.DepartureDate)),
		table.ValueParam("$date_backward", ydbLib.Uint32Value(query.ReturnDate)),
	)
	statement := repository.dynamicByPartnerQuery
	start := time.Now()
	res, err := repository.performRequest(
		ctx,
		statement,
		params,
		repository.dynamicRequestByPartnerTimeout,
		query.JobID,
	)
	metrics.GlobalWizardMetrics().YdbDynamicByPartnerRequestTimer.RecordDuration(time.Since(start))
	if err != nil {
		return nil, err
	}
	return repository.readWizardResults(res, ctx, query.JobID)
}

func (repository *YDBResultsRepository) performRequest(
	ctx context.Context,
	statement string,
	operationParams *table.QueryParameters,
	timeout time.Duration,
	jobID string,
) (*table.Result, error) {
	var (
		res           *table.Result
		retryDoneInfo *table.RetryLoopDoneInfo
	)
	err := repository.retryer.Do(
		table.WithRetryTrace(
			ydbLib.WithOperationTimeout(ctx, timeout),
			table.RetryTrace{
				OnLoop: func(table.RetryLoopStartInfo) func(table.RetryLoopDoneInfo) {
					return func(info table.RetryLoopDoneInfo) {
						retryDoneInfo = &info
					}
				},
			},
		),
		table.OperationFunc(func(ctx context.Context, session *table.Session) (err error) {
			requestSpan, ctx := opentracing.StartSpanFromContext(ctx, "Performing YDB request")
			defer requestSpan.Finish()
			prepareStatementSpan, _ := opentracing.StartSpanFromContext(ctx, "Prepare statement")
			preparationStart := time.Now()
			statement, err := session.Prepare(ctx, statement)
			metrics.GlobalWizardMetrics().YdbStatementPreparationTimer.RecordDuration(time.Since(preparationStart))
			prepareStatementSpan.Finish()
			if err != nil {
				return domain.NewWizardError(
					fmt.Sprintf("couldn't prepare ydb statement: %s", err.Error()),
					domain.YdbStatementPreparationError,
				)
			}
			executeStatementSpan, _ := opentracing.StartSpanFromContext(ctx, "Execute statement")
			defer executeStatementSpan.Finish()
			executionStart := time.Now()
			_, res, err = statement.Execute(
				ctx,
				repository.transactionControl,
				operationParams,
			)
			repository.appLogger.Info(fmt.Sprintf("ydb request execution time: %d", time.Since(executionStart).Milliseconds()), log.String("job_id", jobID))
			metrics.GlobalWizardMetrics().YdbStatementExecutionTimer.RecordDuration(time.Since(executionStart))
			if err != nil {
				return domain.NewWizardError(
					fmt.Sprintf("couldn't execute ydb statement: %s", err.Error()),
					domain.YdbStatementExecutionError,
				)
			}
			return
		}),
	)
	if err != nil {
		repository.appLogger.Warn(fmt.Sprintf("ydb request finished with error: %+v (session pool stats: %+v, retry info: %+v)", err, repository.sessionPool.Stats(), retryDoneInfo))
	}
	return res, err
}

func (repository *YDBResultsRepository) readWizardResults(
	res *table.Result,
	ctx context.Context,
	jobID string,
) ([]*ydbResult.WizardSearchResult, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "Reading YDB results")
	defer span.Finish()
	res.NextSet()
	wizardSearchResults := make([]*ydbResult.WizardSearchResult, 0)
	for res.NextRow() {
		wizardSearchResult := ydbResult.NewWizardSearchResult(ctx)
		start := time.Now()
		if err := wizardSearchResult.Scan(res); err != nil {
			return nil, err
		}
		repository.appLogger.Info(fmt.Sprintf("ydb response scan: %d", time.Since(start).Milliseconds()), log.String("job_id", jobID))
		if wizardSearchResult.SearchResult.Error != nil {
			return nil, wizardSearchResult.SearchResult.Error
		}
		if wizardSearchResult.FilterState.Error != nil {
			return nil, wizardSearchResult.FilterState.Error
		}
		wizardSearchResults = append(wizardSearchResults, wizardSearchResult)
	}
	return wizardSearchResults, nil
}

func (repository *YDBResultsRepository) readWizardResult(
	res *table.Result,
	ctx context.Context,
	jobID string,
) (*ydbResult.WizardSearchResult, error) {
	wizardSearchResults, err := repository.readWizardResults(res, ctx, jobID)
	if err != nil || len(wizardSearchResults) == 0 {
		return nil, err
	}
	fmt.Printf("data: %+v\n", wizardSearchResults[0])
	return wizardSearchResults[0], nil
}

func (repository *YDBResultsRepository) fillSessionPool() error {
	sessionPoolPreparingContext := context.Background()
	createSessionsAttempts := 4 * repository.preparedSessionsCount
	queries := []string{
		repository.minPriceByDirectionQuery,
		repository.minPriceByDirectionWithTTLQuery,
		repository.minPriceByDirectionWithTTLParallelQuery,
		repository.minPriceByDirectionByPartnerQuery,
		repository.staticWithDateQuery,
		repository.staticWithDateAndTTLQuery,
		repository.staticWithDateByPartnerQuery,
		repository.dynamicQuery,
		repository.dynamicWithTTLQuery,
		repository.dynamicByPartnerQuery,
		repository.personalizationDateForwardByPartnerQuery,
	}
	for i := 0; i < repository.preparedSessionsCount && createSessionsAttempts > 0; {
		session, err := repository.sessionPool.Create(sessionPoolPreparingContext)
		if err != nil {
			repository.appLogger.Error(fmt.Sprintf("an error occured during creating session: %s", err.Error()))
			createSessionsAttempts--
			continue
		}
		for _, query := range queries {
			for createSessionsAttempts > 0 {
				if _, err = session.Prepare(sessionPoolPreparingContext, query); err != nil {
					repository.appLogger.Error(fmt.Sprintf("an error occured during preparation query [%s]: %s", query, err.Error()))
					createSessionsAttempts--
				} else {
					break
				}
			}
		}
		if createSessionsAttempts > 0 {
			i++
		}
	}
	if createSessionsAttempts == 0 {
		return fmt.Errorf("couldn't fill YDB sessions pool")
	}
	repository.appLogger.Info(fmt.Sprintf("ydb session pool has been filled with %d sessions", repository.preparedSessionsCount))
	return nil
}

func (repository *YDBResultsRepository) performParallelRequests(
	ctx context.Context,
	query *StaticQuery,
	params *table.QueryParameters,
) (*ydbResult.WizardSearchResult, error) {
	start := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().YdbExperimentalStaticParallelRequestsTimer.RecordDuration(time.Since(start))
	}()
	nearDatesParams := params
	farDatesParams := table.NewQueryParameters()
	nearDatesParams.Each(func(name string, value ydbLib.Value) {
		farDatesParams.Add(table.ValueParam(name, value))
	})
	nearDatesParams.Add(table.ValueParam("$lower_date_forward_bound", ydbLib.Uint32Value(query.NowInDays)))
	nearDatesParams.Add(table.ValueParam(
		"$upper_date_forward_bound",
		ydbLib.Uint32Value(query.NowInDays+uint32(repository.nearDaysCount)),
	))
	nearDatesParams.Add(table.ValueParam("$ttl", ydbLib.Uint64Value(uint64(repository.nearDateTTL))))
	farDatesParams.Add(table.ValueParam(
		"$lower_date_forward_bound",
		ydbLib.Uint32Value(query.NowInDays+uint32(repository.nearDaysCount+1)),
	))
	farDatesParams.Add(table.ValueParam(
		"$upper_date_forward_bound",
		ydbLib.Uint32Value(query.NowInDays+upperDateForwardBound),
	))
	farDatesParams.Add(table.ValueParam("$ttl", ydbLib.Uint64Value(uint64(repository.farDateTTL))))

	wg := sync.WaitGroup{}
	wg.Add(2)
	var nearRes *table.Result
	var nearErr error
	var farRes *table.Result
	var farErr error
	go func() {
		defer wg.Done()
		nearRes, nearErr = repository.performRequest(
			ctx,
			repository.minPriceByDirectionWithTTLParallelQuery,
			nearDatesParams,
			repository.staticRequestTimeout,
			query.JobID,
		)
	}()
	go func() {
		defer wg.Done()
		farRes, farErr = repository.performRequest(
			ctx,
			repository.minPriceByDirectionWithTTLParallelQuery,
			farDatesParams,
			repository.staticRequestTimeout,
			query.JobID,
		)
	}()
	wg.Wait()
	metrics.GlobalWizardMetrics().YdbStaticWithoutDateRequestTimer.RecordDuration(time.Since(start))
	if nearErr != nil && farErr != nil {
		message := fmt.Sprintf("both responses with err: [%+v] [%+v]", nearErr, farErr)
		repository.appLogger.Warn(message)
		return nil, domain.NewWizardError(message, domain.YdbStatementExecutionError)
	}
	if nearErr == nil && farErr != nil {
		return repository.readWizardResult(nearRes, ctx, query.JobID)
	}
	if nearErr != nil && farErr == nil {
		return repository.readWizardResult(farRes, ctx, query.JobID)
	}
	nearWizardResult, nearErr := repository.readWizardResult(nearRes, ctx, query.JobID)
	farWizardResult, farErr := repository.readWizardResult(farRes, ctx, query.JobID)
	if nearErr != nil && farErr != nil {
		message := fmt.Sprintf("couldn't read both responses: [%+v] [%+v]", nearErr, farErr)
		repository.appLogger.Warn(message)
		return nil, domain.NewWizardError(message, domain.YdbBadResponse)
	}
	if nearErr == nil && farErr != nil {
		return nearWizardResult, nil
	}
	if nearErr != nil && farErr == nil {
		return farWizardResult, nil
	}
	if helpers.IsNil(nearWizardResult) && helpers.IsNil(farWizardResult) {
		return nil, nil
	}
	if helpers.IsNil(nearWizardResult) && !helpers.IsNil(farWizardResult) {
		return farWizardResult, nil
	}
	if helpers.IsNil(farWizardResult) && !helpers.IsNil(nearWizardResult) {
		return nearWizardResult, nil
	}
	if nearWizardResult.MinPrice <= farWizardResult.MinPrice {
		return nearWizardResult, nil
	}
	return farWizardResult, nil
}
