package repositories

import (
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/containers"
)

type (
	Settlement interface {
		GetAll() []*models.Settlement
		GetByID(id int) (*models.Settlement, bool)
		GetAnyCodeByID(id int) (string, bool)
		GetByGeoID(int) (*models.Settlement, bool)
		GetByIataCode(string) (*models.Settlement, bool)
		GetAviaRegionCenter(regionID int) (*models.Settlement, bool)
		GetRegionCapital(regionID int) (*models.Settlement, bool)
		GetAviaCapital(countryID int) (*models.Settlement, bool)
		GetCapital(countryID int) (*models.Settlement, bool)
		GetStationIDs(settlementID int) containers.SetOfInt

		IsAviaSettlement(int) bool
	}

	idToSettlementMapper           map[int]*models.Settlement
	geoIDToSettlementMapper        map[int]*models.Settlement
	regionIDToRegionCenterMapper   map[int]*models.Settlement
	regionIDToRegionCapitalMapper  map[int]*models.Settlement
	countryIDToCapitalMapper       map[int]*models.Settlement
	settlementIDToStationIDsMapper map[int]containers.SetOfInt
	iataToSettlementMapper         map[string]*models.Settlement

	SettlementRepository struct {
		settlementsReference *references.Settlement

		idToSettlementMapper    idToSettlementMapper
		geoIDToSettlementMapper geoIDToSettlementMapper
		iataToSettlementMapper  iataToSettlementMapper
		aviaSettlements         containers.SetOfInt

		regionCapitalByRegionID regionIDToRegionCapitalMapper
		capitalByCountryID      countryIDToCapitalMapper

		aviaRegionCenterByRegionID regionIDToRegionCenterMapper
		aviaCapitalByCountryID     countryIDToCapitalMapper

		stationIDsBySettlementID settlementIDToStationIDsMapper
	}
)

func (repository SettlementRepository) GetAll() []*models.Settlement {
	return repository.settlementsReference.GetAll()
}

func (repository SettlementRepository) GetByID(id int) (*models.Settlement, bool) {
	settlement, found := repository.idToSettlementMapper[id]
	return settlement, found
}

func (repository SettlementRepository) GetByGeoID(geoID int) (*models.Settlement, bool) {
	settlement, found := repository.geoIDToSettlementMapper[geoID]
	return settlement, found
}

func (repository SettlementRepository) GetAnyCodeByID(id int) (string, bool) {
	settlement, found := repository.idToSettlementMapper[id]
	if !found {
		return "", false
	}
	if settlement.IataCode != "" {
		return settlement.IataCode, true
	}
	return settlement.SirenaID, true
}

func (repository SettlementRepository) GetByIataCode(iata string) (*models.Settlement, bool) {
	settlement, found := repository.iataToSettlementMapper[iata]
	return settlement, found
}

func (repository SettlementRepository) IsAviaSettlement(id int) bool {
	return repository.aviaSettlements.Contains(id)
}

func (repository SettlementRepository) GetAviaRegionCenter(regionID int) (*models.Settlement, bool) {
	center, found := repository.aviaRegionCenterByRegionID[regionID]
	return center, found
}

func (repository SettlementRepository) GetRegionCapital(regionID int) (*models.Settlement, bool) {
	center, found := repository.regionCapitalByRegionID[regionID]
	return center, found
}

func (repository SettlementRepository) GetAviaCapital(countryID int) (*models.Settlement, bool) {
	center, found := repository.aviaCapitalByCountryID[countryID]
	return center, found
}

func (repository SettlementRepository) GetCapital(countryID int) (*models.Settlement, bool) {
	center, found := repository.capitalByCountryID[countryID]
	return center, found
}

func (repository SettlementRepository) GetStationIDs(settlementID int) containers.SetOfInt {
	if stationIDs, found := repository.stationIDsBySettlementID[settlementID]; found {
		return stationIDs
	}
	return containers.SetOfInt{}
}

func NewSettlementRepository(
	stationRepository Station,
	stationToSettlementReference *references.StationToSettlement,
	settlementsReference *references.Settlement,
) *SettlementRepository {
	repository := &SettlementRepository{
		settlementsReference:       settlementsReference,
		idToSettlementMapper:       make(idToSettlementMapper),
		geoIDToSettlementMapper:    make(geoIDToSettlementMapper),
		iataToSettlementMapper:     make(iataToSettlementMapper),
		aviaRegionCenterByRegionID: make(regionIDToRegionCenterMapper),
		capitalByCountryID:         make(countryIDToCapitalMapper),
		regionCapitalByRegionID:    make(regionIDToRegionCapitalMapper),
		aviaCapitalByCountryID:     make(countryIDToCapitalMapper),
		stationIDsBySettlementID:   make(settlementIDToStationIDsMapper),
		aviaSettlements:            make(containers.SetOfInt),
	}

	const regionCapitalID = 2
	const capitalID = 1

	for _, settlement := range settlementsReference.GetAll() {
		repository.idToSettlementMapper[settlement.ID] = settlement
		repository.geoIDToSettlementMapper[settlement.GeoID] = settlement

		if settlement.IataCode != "" {
			repository.iataToSettlementMapper[settlement.IataCode] = settlement
		}
		if settlement.RegionID > 0 && settlement.MajorityID == regionCapitalID {
			repository.regionCapitalByRegionID[settlement.RegionID] = settlement
		}
		if settlement.CountryID > 0 && settlement.MajorityID == capitalID {
			repository.capitalByCountryID[settlement.CountryID] = settlement
		}
		if settlement.RegionID > 0 && settlement.MajorityID == capitalID {
			if _, ok := repository.regionCapitalByRegionID[settlement.RegionID]; !ok {
				repository.regionCapitalByRegionID[settlement.RegionID] = settlement
			}
		}
	}
	repository.fillAviaSettlements(stationRepository, stationToSettlementReference)

	for id := range repository.aviaSettlements {
		settlement := repository.idToSettlementMapper[id]
		if settlement.RegionID > 0 && settlement.MajorityID == regionCapitalID {
			repository.aviaRegionCenterByRegionID[settlement.RegionID] = settlement
		}
		if settlement.CountryID > 0 && settlement.MajorityID == capitalID {
			repository.aviaCapitalByCountryID[settlement.CountryID] = settlement
		}
	}
	return repository
}

func (repository SettlementRepository) fillAviaSettlements(
	stationRepository Station,
	stationToSettlementReference *references.StationToSettlement,
) {
	for _, station := range stationRepository.GetAll() {
		if !station.IsAirport() {
			continue
		}

		if settlement, found := repository.idToSettlementMapper[station.SettlementID]; found {
			repository.aviaSettlements.Add(settlement.ID)
		}
		if stationIDs, ok := repository.stationIDsBySettlementID[station.SettlementID]; ok {
			stationIDs.Add(station.ID)
		} else {
			repository.stationIDsBySettlementID[station.SettlementID] = containers.SetOfInt{}
			repository.stationIDsBySettlementID[station.SettlementID].Add(station.ID)
		}
	}

	for _, stationToSettlement := range stationToSettlementReference.GetAll() {
		station, found := stationRepository.GetByID(stationToSettlement.StationID)
		if !found {
			continue
		}
		if !station.IsAirport() {
			continue
		}
		if settlement, found := repository.GetByID(stationToSettlement.SettlementID); found {
			repository.aviaSettlements.Add(settlement.ID)

			if stationIDs, ok := repository.stationIDsBySettlementID[settlement.ID]; ok {
				stationIDs.Add(station.ID)
			} else {
				repository.stationIDsBySettlementID[settlement.ID] = containers.SetOfInt{}
				repository.stationIDsBySettlementID[settlement.ID].Add(station.ID)
			}
		}
	}
}
