package repositories

import (
	"strconv"
	"strings"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type (
	Station interface {
		GetAll() []*models.Station
		GetByID(id int) (*models.Station, bool)
		GetByCode(string) []*models.Station
		GetAnyCodeByID(id int) (string, bool)
		GetIATACodeByID(id int) (string, bool)
		GetByPointKey(pointKey string) (*models.Station, bool)
	}

	idToStationMapper    map[int]*models.Station
	idToCodeMapper       map[int]string
	codeToStationsMapper map[string][]*models.Station

	StationRepository struct {
		stationReference     *references.Station
		idToStationMapper    idToStationMapper
		codeToStationsMapper codeToStationsMapper
		idToIATAMapper       idToCodeMapper
		idToSirenaMapper     idToCodeMapper
		idToICAOMapper       idToCodeMapper
	}
)

func (repository StationRepository) GetAll() []*models.Station {
	return repository.stationReference.GetAll()
}

func (repository StationRepository) GetByID(id int) (*models.Station, bool) {
	station, found := repository.idToStationMapper[id]
	return station, found
}

func (repository StationRepository) GetByPointKey(pointKey string) (*models.Station, bool) {
	if len(pointKey) < 2 {
		return nil, false
	}
	if id, err := strconv.Atoi(pointKey[1:]); err == nil {
		if station, found := repository.idToStationMapper[id]; found {
			return station, found
		}
	}
	return nil, false
}

func (repository StationRepository) GetIATACodeByID(id int) (string, bool) {
	iata, found := repository.idToIATAMapper[id]
	return iata, found
}

func (repository StationRepository) GetAnyCodeByID(id int) (string, bool) {
	if iata, ok := repository.idToIATAMapper[id]; ok {
		return iata, true
	}
	if sirena, ok := repository.idToSirenaMapper[id]; ok {
		return sirena, true
	}
	icao, found := repository.idToICAOMapper[id]
	return icao, found
}

func (repository StationRepository) GetByCode(code string) []*models.Station {
	if stations, found := repository.codeToStationsMapper[strings.ToLower(code)]; found {
		return stations
	}
	return make([]*models.Station, 0)
}

func NewStationRepository(
	stationCodeReference *references.StationCode,
	stationReference *references.Station,
	codeSystemReference *references.CodeSystem,
) Station {
	repository := &StationRepository{
		stationReference:     stationReference,
		idToStationMapper:    make(idToStationMapper),
		idToIATAMapper:       make(idToCodeMapper),
		idToSirenaMapper:     make(idToCodeMapper),
		idToICAOMapper:       make(idToCodeMapper),
		codeToStationsMapper: make(codeToStationsMapper),
	}

	for _, station := range stationReference.GetAll() {
		repository.idToStationMapper[station.ID] = station
	}

	if len(repository.idToStationMapper) == 0 {
		return repository
	}

	for _, stationCode := range stationCodeReference.GetAll() {
		if station, found := repository.idToStationMapper[stationCode.StationID]; found {
			code := strings.ToLower(stationCode.Code)
			repository.codeToStationsMapper.ensureKey(code)
			repository.codeToStationsMapper[code] = append(
				repository.codeToStationsMapper[code],
				station,
			)
			if codeSystem, ok := codeSystemReference.GetByID(stationCode.SystemID); ok {
				switch codeSystem.Code {
				case "iata":
					repository.idToIATAMapper[station.ID] = stationCode.Code
				case "sirena":
					repository.idToSirenaMapper[station.ID] = stationCode.Code
				case "icao":
					repository.idToICAOMapper[station.ID] = stationCode.Code
				}
			}
		}
	}
	return repository
}

func (mapper codeToStationsMapper) ensureKey(key string) {
	if _, found := mapper[key]; !found {
		mapper[key] = make([]*models.Station, 0)
	}
}
