package repositories

import (
	"fmt"
	"reflect"

	"github.com/iancoleman/strcase"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
)

type (
	TranslatedTitle interface {
		GetByID(id int) (title *models.TranslatedTitle, found bool)
		GetTitleTranslation(titleID int, lang models.Lang, grammaticalCase models.GrammaticalCase) (string, error)
		GetCrimeaTranslation(lang models.Lang) string
		GetOldTitleTranslation(model interface{}, lang models.Lang) (string, error)
		LocTitleLocativeWithPreposition(titleID int, preposition string, lang models.Lang) *string
	}

	idToTranslatedTitleMapper map[int]*models.TranslatedTitle

	TranslatedTitleRepository struct {
		translatedTitleReference *references.TranslatedTitle

		idToTranslatedTitle   idToTranslatedTitleMapper
		languageFallbacks     map[models.Lang]models.Lang
		languageCaseFallbacks map[models.LanguageCasePair]models.LanguageCasePair
	}
)

func (repository *TranslatedTitleRepository) GetCrimeaTranslation(lang models.Lang) string {
	switch lang {
	case consts.LangRU:
		return "Крым"
	case consts.LangUK:
		return "Крим"
	case consts.LangTR:
		return "Kırım"
	}
	return "Крым"
}

func (repository *TranslatedTitleRepository) GetTitleTranslation(
	titleID int,
	lang models.Lang,
	grammaticalCase models.GrammaticalCase,
) (string, error) {
	if titleID == consts.FakeCrimeaTitleID {
		return repository.GetCrimeaTranslation(lang), nil
	}
	translatedTitle, found := repository.idToTranslatedTitle[titleID]
	if !found {
		return "", fmt.Errorf("no translated title with id %d was not found", titleID)
	}
	if translation, err := translatedTitle.GetTranslation(lang, grammaticalCase); err == nil {
		return translation, nil
	}
	if fallback, ok := repository.languageCaseFallbacks[models.LanguageCasePair{Lang: lang, Case: grammaticalCase}]; ok {
		if translation, err := translatedTitle.GetTranslation(fallback.Lang, fallback.Case); err == nil {
			return translation, nil
		}
	}
	if translation, err := translatedTitle.GetTranslation(lang, consts.CaseNominative); err == nil {
		return translation, nil
	}
	if fallback, ok := repository.languageFallbacks[lang]; ok {
		if translation, err := translatedTitle.GetTranslation(fallback, consts.CaseNominative); err == nil {
			return translation, nil
		}
	}
	return "", fmt.Errorf("no translation for id %d", titleID)
}

func (repository *TranslatedTitleRepository) GetOldTitleTranslation(
	model interface{}, lang models.Lang,
) (string, error) {
	v := reflect.Indirect(reflect.ValueOf(model))
	if helpers.IsNil(model) || v.IsZero() {
		return "", fmt.Errorf("translation: zero value %v", model)
	}

	fieldName := "Title" + strcase.ToCamel(lang.String())
	if value, ok := getField(v, fieldName); ok {
		return value, nil
	}

	if fallback, ok := repository.languageFallbacks[lang]; ok {
		fieldName := "Title" + strcase.ToCamel(fallback.String())
		if value, ok := getField(v, fieldName); ok {
			return value, nil
		}
	}

	if value, ok := getField(v, "Title"); ok {
		return value, nil
	}
	return "", fmt.Errorf("no old translation for %v", model)
}

func getField(value reflect.Value, field string) (string, bool) {
	if _, ok := value.Type().FieldByName(field); !ok {
		return "", false
	}
	f := value.FieldByName(field)
	if !f.IsZero() {
		return f.String(), true
	}

	return "", false
}

func (repository *TranslatedTitleRepository) GetByID(id int) (title *models.TranslatedTitle, found bool) {
	title, found = repository.idToTranslatedTitle[id]
	return title, found
}

func (repository *TranslatedTitleRepository) LocTitleLocativeWithPreposition(titleID int, preposition string, lang models.Lang) *string {
	switch lang {
	case consts.LangRU, consts.LangKK:
		if preposition == "" {
			preposition = "в"
		}
		title, _ := repository.GetTitleTranslation(titleID, lang, consts.CaseLocative)
		return repository.titleWithPreposition(preposition, title)
	case consts.LangTR:
		title, _ := repository.GetTitleTranslation(titleID, lang, consts.CaseNominative)
		return repository.titleWithPreposition("", title)
	case consts.LangEN:
		title, _ := repository.GetTitleTranslation(titleID, lang, consts.CaseNominative)
		return repository.titleWithPreposition("in", title)
	case consts.LangUK:
		title, _ := repository.GetTitleTranslation(titleID, lang, consts.CaseNominative)
		return repository.titleWithPreposition("в", title)
	case consts.LangBE:
		title, _ := repository.GetTitleTranslation(titleID, lang, consts.CaseNominative)
		return repository.titleWithPreposition("у", title)
	}
	return nil
}

func (repository *TranslatedTitleRepository) titleWithPreposition(preposition, title string) *string {
	if len(preposition) > 0 {
		preposition = preposition + consts.NBSP
	} else {
		preposition = ""
	}
	if len(title) > 0 {
		return ptr.String(preposition + title)
	}
	return nil
}

func NewTranslatedTitleRepository(
	translatedTitleReference *references.TranslatedTitle,
	languageFallbacks map[models.Lang]models.Lang,
	languageCaseFallbacks map[models.LanguageCasePair]models.LanguageCasePair,
) TranslatedTitle {
	repository := TranslatedTitleRepository{
		translatedTitleReference: translatedTitleReference,
		idToTranslatedTitle:      make(idToTranslatedTitleMapper),
		languageFallbacks:        languageFallbacks,
		languageCaseFallbacks:    languageCaseFallbacks,
	}
	for _, translatedTitle := range translatedTitleReference.GetAll() {
		repository.idToTranslatedTitle[translatedTitle.ID] = translatedTitle
	}
	return &repository
}
