package ydb

import (
	"bytes"
	"compress/zlib"
	"context"
	"encoding/json"
	"io/ioutil"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	wizardProto "a.yandex-team.ru/travel/proto/avia/wizard"
)

type SearchResultScanner struct {
	Context context.Context
	Value   *wizardProto.SearchResult
	Error   error
}

func (s SearchResultScanner) Get() (string, bool) {
	panic("conversion from SearchResult to protobuf is not implemented")
}

func (s *SearchResultScanner) Set(v string) {
	span, _ := opentracing.StartSpanFromContext(s.Context, "Deserializing SearchResult")
	now := time.Now()
	defer func() {
		metrics.GlobalWizardMetrics().FaresProtobufDeserialization.RecordDuration(time.Since(now))
	}()
	defer span.Finish()
	var searchResultProto wizardProto.SearchResult
	zlibReader, err := zlib.NewReader(bytes.NewReader([]byte(v)))
	var protobufBytes []byte
	if err != nil {
		if err == zlib.ErrHeader || err == zlib.ErrChecksum {
			protobufBytes = []byte(v)
		} else {
			s.Error = err
			return
		}
	} else if protobufBytes, err = ioutil.ReadAll(zlibReader); err != nil {
		s.Error = err
		return
	}
	err = proto.Unmarshal(protobufBytes, &searchResultProto)
	if err != nil {
		s.Error = err
		return
	}
	*s = SearchResultScanner{Value: &searchResultProto}
}

type FilterStateScanner struct {
	Context context.Context
	Value   results.FilterState
	Error   error
}

func (s FilterStateScanner) Get() (string, bool) {
	panic("conversion from FilterState to json is not implemented")
}

func (s *FilterStateScanner) Set(v string) {
	span, _ := opentracing.StartSpanFromContext(s.Context, "Deserializing FilterState")
	defer span.Finish()
	var filterState results.FilterState
	err := json.Unmarshal([]byte(v), &filterState)
	if err != nil {
		*s = FilterStateScanner{Error: err}
		return
	}
	*s = FilterStateScanner{Value: filterState}
}

//go:generate ydbgen -wrap optional

//ydb:gen
//ydb:set wrap:none
type WizardSearchResult struct {
	DateForward  uint32              `ydb:"type:uint32?"`
	DateBackward uint32              `ydb:"type:uint32?"`
	MinPrice     int32               `ydb:"type:int32?"`
	SearchResult SearchResultScanner `ydb:"type:string?"`
	FilterState  FilterStateScanner  `ydb:"type:string?"`
}

func NewWizardSearchResult(ctx context.Context) *WizardSearchResult {
	return &WizardSearchResult{
		SearchResult: SearchResultScanner{Context: ctx},
		FilterState:  FilterStateScanner{Context: ctx},
	}
}
