package dynamicresources

import (
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/headers"
)

type Config struct {
	Addr  string
	Route string
}

var DefaultConfig = Config{
	Addr:  "[::]:9006",
	Route: "/update_dicts",
}

type Service struct {
	logger   log.Logger
	onUpdate func(string) error
	config   Config
}

func NewService(logger log.Logger, config Config, onUpdate func(string) error) *Service {
	return &Service{logger: logger.WithName("DynamicResourcesService"), config: config, onUpdate: onUpdate}
}

func (s *Service) BackroundRun() {
	router := chi.NewRouter()
	router.Get(s.config.Route, s.updateHandler)
	go func() {
		_ = http.ListenAndServe(s.config.Addr, router)
	}()
}

type updateResourcesResponse struct {
	Status string `json:"status"`
	Error  string `json:"error,omitempty"`
}

var successResponse, _ = json.Marshal(updateResourcesResponse{Status: "success"})

func (s *Service) updateHandler(w http.ResponseWriter, r *http.Request) {
	w.WriteHeader(http.StatusOK)
	w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	query := r.URL.Query()
	source := ""
	if sources, present := query["source"]; present {
		source = sources[0]
	}
	if err := s.onUpdate(source); err == nil {
		_, _ = w.Write(successResponse)
	} else {
		s.logger.Error("failed to update dynamic resources", log.Error(err))
		response, _ := json.Marshal(updateResourcesResponse{Status: "failed", Error: err.Error()})
		_, _ = w.Write(response)
	}
}
