package shared

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	libMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/service"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights/models"
)

const (
	serviceName        = "shared_flights"
	flightsMethod      = "flight-range"
	flightsMultiMethod = "flight-range-multi"
)

type Client struct {
	httpClient        *http.Client
	baseURL           string
	appLogger         log.Logger
	flightsMapper     *FlightsMapper
	serviceCallLogger *service.Logger
}

func NewSharedFlightsClient(
	httpClient *http.Client,
	baseURL string,
	appLogger log.Logger,
	serviceCallLogger *service.Logger,
	flightsMapper *FlightsMapper,
) *Client {
	return &Client{
		httpClient:        httpClient,
		baseURL:           baseURL,
		appLogger:         appLogger,
		flightsMapper:     flightsMapper,
		serviceCallLogger: serviceCallLogger,
	}
}

func (client *Client) GetFlights(flightsRequest *models.Request) (flights flights.Flights, err error) {
	handlerSpan, ctx := opentracing.StartSpanFromContext(flightsRequest.Context, "Shared-models request")
	defer handlerSpan.Finish()

	logger := log.With(client.appLogger,
		log.String("job_id", flightsRequest.JobID),
		log.String("method", flightsMethod),
	)

	requestURL := fmt.Sprintf("%s/%s", client.baseURL, buildFlightRequestURL(flightsRequest))
	request, err := http.NewRequest(http.MethodGet, requestURL, nil)
	if err != nil {
		return nil, err
	}

	flightsResponse := Flights{}
	requestDuration, err := client.do(
		ctx, request, logger, &flightsResponse,
		metrics.GlobalWizardMetrics().SharedFlightsGoodRequestTimer,
		metrics.GlobalWizardMetrics().SharedFlightsBadRequestTimer,
	)
	defer func() {
		client.serviceCallLogger.Log(service.NewRecord(flightsRequest.JobID, serviceName, flightsMethod, requestDuration, err != nil))
	}()

	return client.flightsMapper.Map(flightsResponse)
}

func buildFlightRequestURL(request *models.Request) string {
	values := url.Values{}
	values.Set("service", "wizard")
	values.Set("main_reqid", request.MainReqID)
	values.Set("reqid", request.ReqID)
	values.Set("job_id", request.JobID)

	values.Set("arrival_day_period", fmt.Sprintf("%d:%d", request.ArrivedFlightsCount, request.NotArrivedFlightsCount))
	values.Set("time_utc", formatDateTime(request.Date))

	splittedNumber := strings.Split(request.FlightNumber, " ")
	companyIata := splittedNumber[0]
	number := splittedNumber[1]

	return fmt.Sprintf("api/v1/%s/%s/%s/?%s", flightsMethod, companyIata, number, values.Encode())
}

func (client *Client) GetFlightsMulti(flightsRequest *models.MultiRequest) (flights.FlightsByCompanyID, error) {
	handlerSpan, ctx := opentracing.StartSpanFromContext(flightsRequest.Context, "Shared-models multi request")
	defer handlerSpan.Finish()

	logger := log.With(client.appLogger,
		log.String("job_id", flightsRequest.JobID),
		log.String("method", flightsMultiMethod),
	)

	requestURL := fmt.Sprintf("%s/%s", client.baseURL, buildFlightMultiRequestURL(flightsRequest))
	request, err := http.NewRequest(http.MethodGet, requestURL, nil)
	if err != nil {
		return nil, err
	}

	flightsResponse := make(map[int]Flights)
	requestDuration, err := client.do(
		ctx, request, logger, &flightsResponse,
		metrics.GlobalWizardMetrics().SharedFlightsMultiGoodRequestTimer,
		metrics.GlobalWizardMetrics().SharedFlightsMultiBadRequestTimer,
	)
	defer func() {
		client.serviceCallLogger.Log(service.NewRecord(flightsRequest.JobID, serviceName, flightsMultiMethod, requestDuration, err != nil))
	}()

	result := make(flights.FlightsByCompanyID, len(flightsResponse))
	for companyID, rawFlights := range flightsResponse {
		mappedFlights, err := client.flightsMapper.Map(rawFlights)
		if err != nil {
			return nil, err
		}
		result[companyID] = mappedFlights
	}

	return result, nil
}

func buildFlightMultiRequestURL(request *models.MultiRequest) string {
	values := url.Values{}
	values.Set("service", "wizard")
	values.Set("main_reqid", request.MainReqID)
	values.Set("reqid", request.ReqID)
	values.Set("job_id", request.JobID)

	values.Set("arrival_day_period", fmt.Sprintf("%d:%d", request.ArrivedFlightsCount, request.NotArrivedFlightsCount))
	values.Set("time_utc", formatDateTime(request.Date))

	return fmt.Sprintf("api/v1/%s/%d/?%s", flightsMultiMethod, request.FlightNumber, values.Encode())
}

func (client *Client) do(
	ctx context.Context,
	request *http.Request,
	logger log.Logger,
	value interface{},
	goodTimer, badTimer libMetrics.Timer,
) (time.Duration, error) {
	request = request.WithContext(ctx)
	start := time.Now()
	response, err := client.httpClient.Do(request)
	requestDuration := time.Since(start)

	if response != nil {
		if response.StatusCode == http.StatusNotFound {
			goodTimer.RecordDuration(requestDuration)
		} else if response.StatusCode == http.StatusOK {
			badTimer.RecordDuration(requestDuration)
		}
		metrics.GlobalWizardMetrics().GetSharedFlightsHTTPCodeCounter(response.StatusCode).Inc()
	}

	if err != nil {
		if !errors.Is(err, context.DeadlineExceeded) && !errors.Is(err, context.Canceled) {
			logger.Error("an error occurred while getting models", log.Error(err))
		}
		return requestDuration, err
	}
	defer response.Body.Close()
	if response.StatusCode != http.StatusOK {
		err = fmt.Errorf("shared-models has responded with code %d", response.StatusCode)
		logger.Warn(err.Error())
		return requestDuration, err
	}

	if err := json.NewDecoder(response.Body).Decode(value); err != nil {
		return requestDuration, fmt.Errorf("couldn't decode shared-flights response: %w", err)
	}
	return requestDuration, nil
}

func formatDateTime(t time.Time) string {
	return t.UTC().Format("2006-01-02T15:04:05")
}
