package shared

import (
	"fmt"
	"time"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

const flightEventDateTimeFormat = "2006-01-02T15:04:05-07:00"

type (
	FlightsMapper struct {
		pointMapper       *PointMapper
		companyRepository repositories.Company
	}

	PointMapper struct {
		stationRepository  repositories.Station
		locationRepository repositories.CachedLocation
	}
)

func NewFlightsMapper(pointMapper *PointMapper, companyRepository repositories.Company) *FlightsMapper {
	return &FlightsMapper{pointMapper: pointMapper, companyRepository: companyRepository}
}

func (mapper *FlightsMapper) Map(flightsToMap Flights) (flights.Flights, error) {
	result := flights.Flights{}
	for _, f := range flightsToMap {
		mappedFlight, err := mapper.mapFlight(f)
		if err != nil {
			return nil, err
		}
		result = append(result, mappedFlight)
	}
	return result, nil
}

func (mapper *FlightsMapper) mapFlight(flight *Flight) (*flights.Flight, error) {
	departureTerminal := flight.Status.DepartureTerminal
	if departureTerminal == "" {
		departureTerminal = flight.DepartureTerminal
	}
	pointFrom, err := mapper.pointMapper.Map(
		flight.AirportFromID,
		flight.DepartureTimezone,
		flight.DepartureUTC,
		flight.Status.Departure,
		departureTerminal,
		flight.Status.DepartureGate,
		flight.Status.DepartureSource,
	)
	if err != nil {
		return nil, err
	}
	arrivalTerminal := flight.Status.ArrivalTerminal
	if arrivalTerminal == "" {
		arrivalTerminal = flight.ArrivalTerminal
	}
	pointTo, err := mapper.pointMapper.Map(
		flight.AirportToID,
		flight.ArrivalTimezone,
		flight.ArrivalUTC,
		flight.Status.Arrival,
		arrivalTerminal,
		flight.Status.ArrivalGate,
		flight.Status.ArrivalSource,
	)
	if err != nil {
		return nil, err
	}
	company, companyExists := mapper.companyRepository.GetByID(int(flight.AirlineID))
	if !companyExists {
		return nil, fmt.Errorf("couldn't find company by id %d", flight.AirlineID)
	}

	rawUpdatedAt := flight.UpdatedAtUTC
	if flight.Status.UpdatedAtUTC > rawUpdatedAt {
		rawUpdatedAt = flight.Status.UpdatedAtUTC
	}
	updatedAt, err := time.Parse("2006-01-02 15:04:05", rawUpdatedAt)
	if err != nil {
		return nil, fmt.Errorf("couldn't parse flight updatedAt: %v", err)
	}

	return &flights.Flight{
		PointFrom: *pointFrom,
		PointTo:   *pointTo,
		Number:    flight.Title,
		Company:   *company,
		StatusInfo: flights.StatusInfo{
			Status:              flight.Status.Status,
			DepartureIsKnown:    flight.Status.Departure != "",
			ArrivalIsKnown:      flight.Status.Arrival != "",
			BaggageCarousels:    flight.Status.BaggageCarousels,
			CheckInDesks:        flight.Status.CheckInDesks,
			Diverted:            flight.Status.Diverted,
			DivertedAirportIata: flight.Status.DivertedAirportIata,
			DivertedAirportID:   flight.Status.DivertedAirportID,
		},
		UpdatedAt: updatedAt,
	}, nil
}

func NewPointMapper(stationRepository repositories.Station, locationRepository repositories.CachedLocation) *PointMapper {
	return &PointMapper{stationRepository: stationRepository, locationRepository: locationRepository}
}

func (mapper *PointMapper) Map(
	airportID int32,
	timezoneName,
	scheduledTime,
	actualTime,
	terminal,
	gate,
	dataSource string,
) (*flights.Point, error) {
	timezone, err := mapper.locationRepository.LoadLocation(timezoneName)
	if err != nil {
		return nil, err
	}
	scheduled, err := helpers.ParseTimeInLocation(scheduledTime, time.UTC)
	if err != nil {
		return nil, err
	}
	airport, airportExists := mapper.stationRepository.GetByID(int(airportID))
	if !airportExists {
		return nil, fmt.Errorf("couldn't find airport by id %d", airportID)
	}
	var parsedActualTime *time.Time
	if actualTime, err := helpers.ParseTimeInLocation(actualTime, timezone); err == nil {
		parsedActualTime = &actualTime
	}
	airportCode, _ := mapper.stationRepository.GetAnyCodeByID(airport.ID)

	return &flights.Point{
		Timezone:      timezone,
		Airport:       airport,
		AirportCode:   airportCode,
		ScheduledTime: scheduled.In(timezone),
		Time:          parsedActualTime,
		Terminal:      helpers.OptionalString(terminal),
		Gate:          helpers.OptionalString(gate),
		IsActual:      parsedActualTime != nil,
		DataSource:    dataSource,
	}, nil
}
