package personalization

import (
	"context"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/balancer"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	personalsearch "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/service"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
	hcbalancer "a.yandex-team.ru/travel/library/go/grpcutil/client/healthcheck_balancer"
	"a.yandex-team.ru/travel/library/go/grpcutil/client/ypresolver"
)

type Client interface {
	SendGetPersonalSearch(ctx context.Context, yandexUID, geoID, jobID string) (response *personalsearch.TGetPersonalSearchResponseV2, personalizationErr error)
}

type GrpcPersonalizationClient struct {
	tvmInterceptor         grpc.UnaryClientInterceptor
	appLogger              log.Logger
	personalizationTimeout time.Duration
	serviceCallLogger      *service.Logger
	grpcClient             personalsearch.PersonalizationServiceV2Client
}

func init() {
	personalizationHealthCheckServiceName := "personalization-api"
	logger, _ := zap.NewDeployLogger(log.DebugLevel)
	balancer.Register(
		hcbalancer.NewBalancerBuilder(
			hcbalancer.WithLogger(logger.WithName("PersonalizationBalancer")),
			hcbalancer.WithHealthCheckServiceName(personalizationHealthCheckServiceName),
			hcbalancer.WithHealthCheckInterval(2000*time.Millisecond),
			hcbalancer.WithBalancingMethod(hcbalancer.BalancingMethodChooseClosest),
		),
	)
}

func CreateConnection(settings *settings.PersonalizationSettings, logger log.Logger) (grpc.ClientConnInterface, error) {
	if !settings.Enabled {
		return nil, nil
	}
	ctx := context.Background()
	ctx, cancel := context.WithTimeout(ctx, 90*time.Second)
	defer cancel()
	resolver := ypresolver.NewYPResolverBuilder(ypresolver.WithLogger(logger.Structured()))
	serviceConfig := hcbalancer.ServiceConfig{
		LoadBalancingPolicy: "healthcheck_balancer",
		LoadBalancingConfig: hcbalancer.NewLoadBalancingConfig(
			hcbalancer.NewLoadBalancerConfig("healthcheck_balancer", settings.ConnectionsPerHost),
		),
	}
	return grpc.DialContext(
		ctx,
		ypresolver.BuildServiceFQDN(settings.YPlannerID),
		grpc.WithResolvers(resolver),
		grpc.WithDefaultServiceConfig(serviceConfig.MarshalToJSON()),
		grpc.WithInsecure(),
		grpc.WithBlock(),
	)
}

func NewPersonalizationClient(
	connection grpc.ClientConnInterface,
	personalizationTimeout time.Duration,
	appLogger log.Logger,
	serviceCallLogger *service.Logger,
) *GrpcPersonalizationClient {
	grpcClient := personalsearch.NewPersonalizationServiceV2Client(connection)
	return &GrpcPersonalizationClient{
		appLogger:              appLogger,
		personalizationTimeout: personalizationTimeout,
		serviceCallLogger:      serviceCallLogger,
		grpcClient:             grpcClient,
	}
}

func (pc *GrpcPersonalizationClient) SendGetPersonalSearch(ctx context.Context, yandexUID, geoID, jobID string) (
	response *personalsearch.TGetPersonalSearchResponseV2, personalizationErr error) {
	ctx, cancel := context.WithTimeout(ctx, pc.personalizationTimeout)
	defer cancel()

	executionStart := time.Now()
	req := personalsearch.TGetAviaHistoryRequestV2{YandexUid: yandexUID}

	response, err := pc.grpcClient.GetAviaHistory(ctx, &req)

	metrics.GlobalWizardMetrics().PersonalizationRequestTimer.RecordDuration(time.Since(executionStart))
	pc.appLogger.Info(fmt.Sprintf("personalization request execution time: %d", time.Since(executionStart).Milliseconds()), log.String("job_id", jobID))
	if err != nil {
		personalizationErr = xerrors.Errorf("personalization app returned an error: %w", err)
		return
	}
	return
}
